
#include <io.h>
#include "CommonFunctions.h"
#include "StreamInfo.h"

void CStreamInfo::Reset()
{
	Packed = false;
	MainMetadataStream = false;
	HaveSize = false;
	SizeOffset = 0;
	SizeBytes = 0;
	ContentOffset = 0;
	FullSize = 0;
	Size = 0;
	LastZeroBytes = 0;
}

CStreamInfo::CStreamInfo()
{
	Reset();
}

void CStreamInfo::UpdateInfo(LPCSTR StreamName)
{
	Reset();

	if( 0 == stricmp("MD Programm text", StreamName) ) Packed = true;
	else if( 0 == stricmp("Inplace description", StreamName) ) Packed = true;

	if( 0 == stricmp("Dialog Stream", StreamName) ) HaveSize = true;
	else if( 0 == stricmp("Main MetaData Stream", StreamName) ) 
	{
		HaveSize = true;
		MainMetadataStream = true;
	}
}

void CStreamInfo::UpdateInfo(const wchar_t* wStreamName)
{
	int len = wcslen(wStreamName) + 1;
	char *StreamName = new char[len];

	wcstombs(StreamName, wStreamName, len);
	UpdateInfo(StreamName);
	delete[] StreamName;
}

void CStreamInfo::DetermineSizeFormat(unsigned long FullSize)
{
	if( !HaveSize ) return;

	if( FullSize == 0 )
	{
		SizeOffset = 0;
		SizeBytes = 1;
		Size = 0;
	}
	else
	{
		if( (FullSize-1) <= 0xFF )
		{
			SizeOffset = 0;
			SizeBytes = 1;
		}
		else if( (FullSize-3) <= 0xFFFF )
		{
			SizeOffset = 1;
			SizeBytes = 2;
		}
		else
		{
			SizeOffset = 3;
			SizeBytes = 4;
		}
	}

	ContentOffset = SizeOffset + SizeBytes;
}

void CStreamInfo::DetermineFileSizeFormat(unsigned long FullSize)
{
	if( !HaveSize ) return;

	if( FullSize == 0 )
	{
		SizeOffset = 0;
		SizeBytes = 1;
		Size = 0;
	}
	else
	{
		if( (FullSize+1) <= 0xFF )
		{
			SizeOffset = 0;
			SizeBytes = 1;
		}
		else if( (FullSize+3) <= 0xFFFF )
		{
			SizeOffset = 1;
			SizeBytes = 2;
		}
		else
		{
			SizeOffset = 3;
			SizeBytes = 4;
		}
	}

	ContentOffset = SizeOffset + SizeBytes;
}

unsigned long CStreamInfo::ReadSizeOfCleanStream(IStream* pStream)
{
	LARGE_INTEGER Offset = {0, 0};
	ULARGE_INTEGER EndPos = {0, 0};

	Offset.LowPart = 0;
	pStream->Seek(Offset, STREAM_SEEK_END, &EndPos);
	FullSize = EndPos.LowPart;
	pStream->Seek(Offset, STREAM_SEEK_SET, NULL);

	Size = FullSize;
	ContentOffset = 0;

	//   
	Offset.LowPart = 0;
	pStream->Seek(Offset, STREAM_SEEK_SET, NULL);

	return Size;
}

unsigned long CStreamInfo::ReadSize(IStream* pStream)
{
	LARGE_INTEGER Offset = {0, 0};
	ULARGE_INTEGER EndPos = {0, 0};

	Offset.LowPart = 0;
	pStream->Seek(Offset, STREAM_SEEK_END, &EndPos);
	FullSize = EndPos.LowPart;
	pStream->Seek(Offset, STREAM_SEEK_SET, NULL);

	SizeOffset = 0;
	SizeBytes = 0;
	ContentOffset = 0;
	if( Packed ) HaveSize = false;

	if( HaveSize )
	{
		DetermineSizeFormat(FullSize);
		if( FullSize > 0 )
		{
			Offset.LowPart = SizeOffset;
			pStream->Seek(Offset, STREAM_SEEK_SET, NULL);
			switch( SizeBytes )
			{
				case 1:
					BYTE bSize;
					pStream->Read(&bSize, SizeBytes, NULL);
					Size = bSize;
					break;
				case 2:
					WORD wSize;
					pStream->Read(&wSize, SizeBytes, NULL);
					Size = wSize;
					break;
				case 4:
					DWORD dwSize;
					pStream->Read(&dwSize, SizeBytes, NULL);
					Size = dwSize;
					break;
				default:
					STATSTG StatSgts;
					pStream->Stat(&StatSgts, STATFLAG_DEFAULT);
					Msg(0, "ERR: Don't know how to read size of stream '%S' (%i bytes)\n", StatSgts.pwcsName, (int)SizeBytes);
					exit(1);
					break;
			}

			CountLastZeroBytes(pStream);
		}
	}
	else
	{
		Size = FullSize;
		ContentOffset = 0;
	}

	//   
	Offset.LowPart = 0;
	pStream->Seek(Offset, STREAM_SEEK_SET, NULL);

	return Size;
}

unsigned long CStreamInfo::ReadFileSize(FILE* File)
{
	SizeOffset = 0;
	SizeBytes = 0;
	ContentOffset = 0;

	fseek(File, 0, SEEK_END);
	FullSize = ftell(File);
	fseek(File, 0, SEEK_SET);

	Size = FullSize;

	DetermineFileSizeFormat(FullSize);
	ContentOffset = SizeOffset + SizeBytes;

	if( FullSize > 0 )
	{
		CountLastZeroBytes(File);
		Size -= LastZeroBytes;
	}

	return Size;
}


unsigned long CStreamInfo::ReadSize(LPCSTR lpStr)
{
	FullSize = strlen(lpStr) + 1; // '\0'  
	Size = FullSize;
	DetermineSizeFormat(FullSize);
	LastZeroBytes = 1;
	Size -= LastZeroBytes;

	return Size;
}

unsigned long CStreamInfo::CountLastZeroBytes(IStream* pStream)
{
	LARGE_INTEGER Offset = {0, 0};
	BYTE byte = 0;
	HRESULT hr;

	Offset.LowPart = FullSize;
	do
	{
		Offset.LowPart--;
		hr = pStream->Seek(Offset, STREAM_SEEK_SET, NULL);
		if( hr != S_OK ) break;
		hr = pStream->Read(&byte, 1, NULL);
		if( hr != S_OK ) break;
		
		if( byte == 0 )
			LastZeroBytes++;
	} while( byte == 0 );

	return LastZeroBytes;
}

unsigned long CStreamInfo::CountLastZeroBytes(FILE* File)
{
	long Offset = FullSize;
	BYTE byte = 0;
	do
	{
		Offset--;
		fseek(File, Offset, SEEK_SET);
		fread(&byte, 1, 1, File);
		if( byte == 0 ) LastZeroBytes++;
	} while( byte == 0 );

	rewind(File);
	return LastZeroBytes;
}
