
#include "Declarations.h"
#include "CommonFunctions.h"
#include "DialogForm.h"
#include "DlgFormFlags.h"
#include "MMS.h"
#include "Storage.h"
#include "Strings.h"


bool ReadBool(const char* str)
{
	if( strcmp(str, "1") == 0 ) return true;
	if( _stricoll(str, "") == 0 ) return true;
	return false;
}


void add_flags(CString& String, int level, const char* id, unsigned int Flags, char* FlagNames[])
{
	CString flags;
	bool delimit = false;
	int bit = 0;
	unsigned int bitmask = 1;

	while( bit < 32 )
	{
		if( Flags & bitmask )
		{
			if( delimit )
				flags += ", ";
			flags += FlagNames[bit];
			delimit = true;
		}
		bit++;
		bitmask <<= 1;
	}

	add_parameter(String, level, id, flags, "");
}

int find_string(char* StringList[], int nStrings, const char* str)
{
	int i = 0;
	while( i < nStrings )
	{
		if( _stricoll(str, StringList[i]) == 0 ) return i;
		i++;
	}
	return -1;
}

unsigned int read_flags(char* value, char* FlagList[])
{
	if( value == NULL ) return 0;

	unsigned int Flags = 0;
	int bit;
	char *token;

	token = strtok(value, ", \t");
	while( NULL != token )
	{
		bit = find_string(FlagList, 32, token);
		if( bit < 0 )
		{
			Msg(0, "ERR:   '%s'\n", token);
			exit(1);
		}
		Flags |= 1 << bit;
		token = strtok(NULL, ", \t");
	}

	return Flags;
}


/////////////////////////////////////////////////////////////////
//////  CPropertySet  //////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
CPropertySet::CPropertySet(CObjectProperty* Props, void* Defaults)
{
	Propertys = Props;
	DefaultValues = Defaults;

	// 
	nProps = 0;
	MaxMMS_Index = 0;
	CObjectProperty* prop = Props;
	int offset = 0;
	while( prop->Size > 0 )
	{
		prop->Offset = offset;
		offset += prop->Size;
		prop++;
		nProps++;
		MaxMMS_Index = max(MaxMMS_Index, prop->MMS_Index);
	}
}

CPropertySet::~CPropertySet()
{
	if( DefaultValues != NULL ) delete DefaultValues;
}
	
int CPropertySet::FindByName(const char* Name)
{
	CObjectProperty* prop = Propertys;
	int i = 0;
	while( prop->Size > 0 )
	{
		if( _stricoll(prop->Name, Name) ==0 )
			return i;
		prop++;
		i++;
	}

	return -1;
}

int CPropertySet::FindByMMSIndex(int Index)
{
	CObjectProperty* prop = Propertys;
	int i = 0;
	while( prop->Size > 0 )
	{
		if( prop->MMS_Index == Index )
			return i;
		prop++;
		i++;
	}
	return -1;
}

void CPropertySet::SetValue(void* Object, int i, CString& Value)
{
	CObjectProperty& prop = Propertys[i];
	switch( prop.Type )
	{
	case CObjectProperty::t_string:
		{
		CString* str_prop_val = (CString*) ((char*)Object + prop.Offset );
		*str_prop_val = Value;
		}
		break;
	case CObjectProperty::t_int:
		{
		int* int_prop_val = (int*) ((char*)Object + prop.Offset);
		*int_prop_val = atoi(Value);
		}
		break;
	case CObjectProperty::t_uint:
		{
		unsigned int* uint_prop_val = (unsigned int*) ((char*)Object + prop.Offset);
		char *stopstring;
		*uint_prop_val = strtol(Value, &stopstring, 0); //Undersatnd 0xNNNN format
		}
		break;
	case CObjectProperty::t_bitset:
		{
		unsigned int* uint_prop_val = (unsigned int*) ((char*)Object + prop.Offset);
		char *stopstring;
		*uint_prop_val = strtol(Value, &stopstring, 0);
		if( (LPCSTR)Value == stopstring )
			*uint_prop_val = read_flags((char*)(LPCSTR)Value, prop.FlagList);
		}
		break;
	case CObjectProperty::t_bool:
		{
		bool* bool_prop_val = (bool*) ((char*)Object + prop.Offset);
		*bool_prop_val = ReadBool(Value);
		}
		break;
	default:
		Msg(0, "ERR:    %i!\n", (int)prop.Type);
		exit(1);
		break;
	}
}

void CPropertySet::BeautyString(CString& String, int level, const void* Object, int i)
{
	CObjectProperty& prop = Propertys[i];
	switch( prop.Type )
	{
	case CObjectProperty::t_string:
		{
		CString* prop_val = (CString*) ((char*)Object + prop.Offset );
		CString* def_val  = (CString*) ((char*)DefaultValues + prop.Offset );
		add_parameter(String, level, prop.Name, *prop_val, *def_val);
		}
		break;
	case CObjectProperty::t_int:
		{
		int* prop_val = (int*) ((char*)Object + prop.Offset);
		int* def_val  = (int*) ((char*)DefaultValues + prop.Offset );
		add_parameter(String, level, prop.Name, *prop_val, *def_val);
		}
		break;
	case CObjectProperty::t_uint:
		{
		unsigned int* prop_val = (unsigned int*) ((char*)Object + prop.Offset);
		unsigned int* def_val  = (unsigned int*) ((char*)DefaultValues + prop.Offset );
		add_parameter(String, level, prop.Name, *prop_val, *def_val);
		}
		break;
	case CObjectProperty::t_bitset:
		{
		unsigned int* prop_val = (unsigned int*) ((char*)Object + prop.Offset);
		unsigned int* def_val  = (unsigned int*) ((char*)DefaultValues + prop.Offset );
		add_flags(String, level, prop.Name, *prop_val, prop.FlagList);
		}
		break;
	case CObjectProperty::t_bool:
		{
		bool* prop_val = (bool*) ((char*)Object + prop.Offset);
		bool* def_val  = (bool*) ((char*)DefaultValues + prop.Offset );
		add_parameter(String, level, prop.Name, *prop_val, *def_val);
		}
		break;
	default:
		Msg(0, "ERR:    %i (%s)!\n", (int)prop.Type, prop.Name);
		exit(1);
		break;
	}
}

void CPropertySet::ToString(CString& String, const void* Object, int i, bool AddComma)
{
	CObjectProperty& prop = Propertys[i];
	switch( prop.Type )
	{
	case CObjectProperty::t_string:
		{
		CString* prop_val = (CString*) ((char*)Object + prop.Offset );
		add_quoted(String, *prop_val, AddComma);
		}
		break;
	case CObjectProperty::t_int:
		{
		int* prop_val = (int*) ((char*)Object + prop.Offset);
		add_quoted(String, *prop_val, AddComma);
		}
		break;
	case CObjectProperty::t_uint:
	case CObjectProperty::t_bitset:
		{
		unsigned int* prop_val = (unsigned int*) ((char*)Object + prop.Offset);
		add_quoted(String, *prop_val, AddComma);
		}
		break;
	case CObjectProperty::t_bool:
		{
		bool* prop_val = (bool*) ((char*)Object + prop.Offset);
		add_quoted(String, *prop_val, AddComma);
		}
		break;
	default:
		Msg(0, "ERR:    %i!\n", (int)prop.Type);
		exit(1);
		break;
	}
}



/////////////////////////////////////////////////////////////////
//////  CFormFont  //////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
CFormFont::CFormFont()
{
	FontSize = 8;
	Param2 = 0;
	Param3 = 0;
	Param4 = 0;
	FontBold = false;
	FontItalic = false;
	FontUnderline = false;
	Param8 = 0;
	Param9 = 0;
	Param10 = 0;
	Param11 = 0;
	Param12 = 0;
	Param13 = 0;
	FontName = "MS Sans Serif";
}

int CFormFont::RealFontSize()
{
	/*      1
	0    =  0
	1    = -1
	       -2
	2.5  = -3
	3    = -4
	4    = -5
	       -6
	5.5  = -7
	6    = -8
	7    = -9
	       -10
	8.5  = -11
	9    = -12
	10   = -13
	       -14
	11.5 = -15
	12   = -16
	*/
	int sz = -FontSize;
	if( sz <= 0 )
		return 0;

	int add = sz % 4;
	if( add >= 2 ) add = 2;
	return add + ((sz / 4) * 3);
}

void CFormFont::ReadRealFontSize(int size)
{
	int add = (size % 3);
	if( add == 2 ) add++;
	FontSize = - ((size / 3) * 4 + add);
}

int CFormFont::FromMMSObject(CMMSObject* Object, int StartIndex)
{
	int MaxIndex = StartIndex;
	int OffsetBold = (char*)&FontBold - (char*)this;

	for( int i = 0; i < PropSet.nProps; i++ )
	{
		int MMS_Index = StartIndex + Props[i].MMS_Index;
		CMMSObject* Value;
		if( MMS_Index == -1 ) //-1 ,      ,        sID
			Value = Object;
		else
			Value = Object->GetPropByNum(MMS_Index);

		if( Props[i].Offset == OffsetBold ) // Bold  
			FontBold = Value->sID == "700" ? true : false;
		else
			PropSet.SetValue(this, i, Value->sID);

		MaxIndex = max(MaxIndex, MMS_Index);
	}

	FontSize = RealFontSize();

	return MaxIndex + 1;
}

void CFormFont::BeautyString(CString& String, int level)
{
	add_parameter(String, level, "", FontName, "");
	add_offset(String, level);
	String += "{\r\n";
	
	level++;

	int OffsetName = (char*)&FontName - (char*)this;
	for( int i = 0; i < PropSet.nProps; i++ )
	{
		if( Props[i].Offset == OffsetName ) //    
			continue;

		PropSet.BeautyString(String, level, this, i);
	}

	add_offset(String, level-1);
	String += "}\r\n";
}


/////////////////////////////////////////////////////////////////
//////  CFormLayers  ////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
void CFormLayers::FromMMSObject(CMMSObject* Object)
{
	ActiveLayer = Object->ID;

	int sz = Object->GetNProps();
	SetSize(sz);
	for( int i = 0; i < sz; i++ )
	{
		CMMSObject* mms_layer = Object->GetPropByNum(i);
		CFormLayer& Layer = operator[](i);

		Layer.Caption = mms_layer->sID;
		Layer.IsVisible = ReadBool( mms_layer->GetPropByNum(0)->sID );
	}
}

void CFormLayers::BeautyString(CString& String)
{
	add_offset(String, 1);
	String += "\r\n";
	add_offset(String, 1);
	String += "{\r\n";

	int sz = GetSize();
	for( int i = 0; i < sz; i++ )
	{
		CFormLayer& Layer = operator[](i);
		char* descr = "";
		if( Layer.IsVisible && i == ActiveLayer )
			descr = ", ";
		else if( Layer.IsVisible )
			descr = "";
		add_parameter(String, 2, Layer.Caption, descr, "", true);
	}

	add_offset(String, 1);
	String += "}\r\n";
}


/////////////////////////////////////////////////////////////////
//////  CFormControlBase  ///////////////////////////////////////
/////////////////////////////////////////////////////////////////
const char* CFormControlBase::ToRealType(const char* type)
{
	switch( type[0] )
	{
		case 'U': return "";
		case 'N': return "";
		case 'S': return "";
		case 'D': return "";
		case 'E': return "";
		case 'B': return "";
		case 'O': return "";
		case 'T': return "";
		case 'K': return "";
	}

	int nType = atoi(type);
	switch( nType )
	{
	case  0: return "";
	case  1: return "";
	case  2: return "";
	case  3: return "";
	case 10: return "";
	case 11: return "";
	case 12: return "";
	case 15: return "";
	case 16: return "";
	}

	return type;
}

CString CFormControlBase::FromRealType(const char* type, bool AsNumber)
{
	int nType = 0;

	if( *type == '\0' )
		nType = 0;
	else if( _stricoll(type, "") == 0)
		nType = 0;
	else if( _stricoll(type, "") == 0 )
		nType = 1;
	else if( _stricoll(type, "") == 0 )
		nType = 2;
	else if( _stricoll(type, "") == 0 )
		nType = 3;
	else if( _stricoll(type, "") == 0 )
		nType = 10;
	else if( _stricoll(type, "") == 0 )
		nType = 11;
	else if( _stricoll(type, "") == 0 )
		nType = 12;
	else if( _stricoll(type, "") == 0 )
		nType = 15;
	else if( _stricoll(type, "") == 0 )
		nType = 16;
	else
		return type;

	if( AsNumber )
	{
		CString ValueType;
		ValueType.Format("%i", nType);
		return ValueType;
	}

	switch( nType )
	{
		case 0:  return "U"; break;
		case 1:  return "N"; break;
		case 2:  return "S"; break;
		case 3:  return "D"; break;
		case 10: return "E"; break;
		case 11: return "B"; break;
		case 12: return "O"; break;
		case 15: return "T"; break;
		case 16: return "K"; break;
		default: return type;
	}
}


/////////////////////////////////////////////////////////////////
//////  CFormControls  //////////////////////////////////////////
/////////////////////////////////////////////////////////////////
void CHotKey::FromMMSObject(CMMSObject* mms_hotkey)
{
	char* ptr = (char*)(LPCSTR)(mms_hotkey->sID) + 3;

	Modifier = strtol(ptr, &ptr, 10);
	KeyCode  = strtol(ptr+5, &ptr, 10);
}

void CHotKey::FromString(CString& String, CLexer& Lexer)
{
	CString mod, key;

	mod = String.SpanExcluding("+-");
	if( mod.GetLength() == String.GetLength() )
	{
		key = mod;
		mod = "";
	}
	else
		key = String.Mid(mod.GetLength() + 1);

	Modifier = atoi(mod);
	KeyCode = atoi(key);

	mod.MakeUpper();
	if( Modifier == 0 && !mod.IsEmpty() )
	{
		if( strstr(mod, "SHIFT") != NULL ) Modifier |= 0x07;
		if( strstr(mod, "CTRL")  != NULL ) Modifier |= 0x0b;
		if( strstr(mod, "ALT")   != NULL ) Modifier |= 0x13;
	}
	
	if( KeyCode == 0 && !key.IsEmpty() )
	{
		if( key.GetLength() == 1 )
		{
			key.MakeUpper();
			KeyCode = (unsigned char)(key[0]);
		}
		else
		{
			for( int i = 0; virtual_keys[i].code != 0; i++ )
			{
				if( stricmp(virtual_keys[i].name, key) != 0 ) continue;
				KeyCode = virtual_keys[i].code;
				i = -1;
				break;
			}
			if( i > 0 )
			{
				Msg(0, "ERR: Unknown key '%s' (%s)\n", key, Lexer.FilePosInfo());
				exit(1);
			}
		}
	}
}

void CHotKey::BeautyString(CString& String, int level)
{
	if( Modifier == 0 && KeyCode == 0 )
		return;

	CString str;
	/*
	  111 - Shift
	 1011 - Ctrl
	10011 - Alt
	*/
	if( Modifier & 0x04 ) str += "Shift";
	if( Modifier & 0x08 ) str += "Ctrl";
	if( Modifier & 0x10 ) str += "Alt";
	if( !str.IsEmpty() ) str += "+";

	if( ('0' <= KeyCode && KeyCode <= '9') || ('A' <= KeyCode && KeyCode <= 'Z') )
		str += (char)KeyCode;
	else
	{
		for( int i = 0; virtual_keys[i].code != 0; i++ )
		{
			if( KeyCode != virtual_keys[i].code ) continue;
			str += virtual_keys[i].name;
			i  = -1;
			break;
		}

		if( i > 0 )
		{
			CString strCode;
			strCode.Format("%i", KeyCode);
			str += strCode;
		}
	}

	add_parameter(String, level, "HotKey", str, "");
}


CFormControl::CFormControl()
{
	Flags = 0;
	PosX = PosY = 0;
	Width = Height = 0;
	IsObjectAttribute = false; //control     -  . 1 - , 0 - 
	Param9 = 0;
	TabOrder = 0;
	ObjectID = 0; //ID  -.    == -1.
	ValueType = "U";
	ValueLen = 0;    //
	ValuePrec = 0;   //
	TypeObectID = 0; //ID    ValueType in [B, O, T, K].    - 0
	ValueFlags = 0;  //1 - , 2 -  .   OR.
	BasicFlags = 0;
	Param24 = 0;
	FontColor = -1; //-1 - default.  hex-: 0xBBGGRR
	Param40 = -1;
	PictureID = 0;
}

CFormColumn::CFormColumn()
{
	ColumnType = 0;
	Width = 0;
	TabOrder = 0;
	ObjectID = 0; //ID  -.    == -1.
	ValueType = "U";
	ValueLen = 0;    //
	ValuePrec = 0;   //
	TypeObectID = 0; //ID    ValueType in [B, O, T, K].    - 0
	ValueFlags = 0;  //1 - , 2 -  .   OR.
	TextAlignment = 0;
	BasicFlags = 0;
}

void CFormControl::FromMMSObject(CMMSObject* mms_control)
{
	int sz = mms_control->GetNProps();
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetHotKey = (char*)&HotKey - (char*)this;

	for( int i = 0; i < PropSet.nProps; i++ )
	{
		int MMS_Index = Props[i].MMS_Index;
		
		if( MMS_Index >= sz )
		{
			Msg(0, "INFO:   '%s' .   ,  gcomp   -vv\n", Props[i].Name);
			continue;
		}

		CMMSObject* Value;
		if( MMS_Index == -1 ) //-1 ,      ,        sID
			Value = mms_control;
		else
			Value = mms_control->GetPropByNum(MMS_Index);

		if( Props[i].Offset == OffsetFont )
			Font.FromMMSObject(mms_control, MMS_Index);
		else if( Props[i].Offset == OffsetHotKey )
			HotKey.FromMMSObject(Value);
		else
			PropSet.SetValue(this, i, Value->sID);
	}

	ValueType = ToRealType(ValueType);
	//  .      .
	quote_linefeeds(LongHint);
	quote_linefeeds(ShortHint);
}

void CFormControl::BeautyString(CString& String, int level)
{
	add_parameter(String, level, Class, Identifier, "", true);
	add_offset(String, level);
	String += "{\r\n";

	level++;

	int OffsetClass = (char*)&Class - (char*)this;
	int OffsetIdentifier = (char*)&Identifier - (char*)this;
	int OffsetType = (char*)&ValueType - (char*)this;
	int OffsetMask = (char*)&Mask - (char*)this;
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetHotKey = (char*)&HotKey - (char*)this;

	for( int i = 0; i < PropSet.nProps; i++ )
	{
		if( Props[i].Offset == OffsetClass ) continue;
		if( Props[i].Offset == OffsetIdentifier ) continue;

		if( Props[i].Offset == OffsetMask ) //      
		{
			if( ValueType == "" || ValueType == "" || ValueType == "11" )
				add_parameter(String, level, "", Mask, "");
			else
				add_parameter(String, level, "", Mask, "");
		}
		else if( Props[i].Offset == OffsetFont && NeedFont() )
			Font.BeautyString(String, level);
		else if( Props[i].Offset == OffsetHotKey )
			HotKey.BeautyString(String, level);
		else if( Props[i].Type != CObjectProperty::t_object )
		{
			if( Props[i].Offset == OffsetType && !NeedType() ) continue;
			PropSet.BeautyString(String, level, this, i);
		}
	}

	level--;

	add_offset(String, level);
	String += "}\r\n";

	if( Param40 > 0 ) Msg(0, "Param40 = %i\n", Param40);
}

bool CFormControl::NeedFont()
{
	if( strcmp(Class, "STATIC") == 0 ) return true;
	return false;
}
bool CFormControl::NeedType()
{
	if( strcmp(Class, "STATIC") == 0 ) return false;
	if( strcmp(Class, "BUTTON") == 0 ) return false;
	if( strcmp(Class, "COMBOBOX") == 0 ) return false;
	if( strcmp(Class, "1CGROUPBOX") == 0 ) return false;
	return true;
}

void CFormColumn::FromMMSObject(CMMSObject* Object)
{
	ColumnType = Object->ID;
	Caption = Object->sID;

	int sz = Object->GetNProps();
	for( int i = 0; i < PropSet.nProps; i++ )
	{
		int MMS_Index = Props[i].MMS_Index;
		if( MMS_Index == -1 ) continue;
		
		if( MMS_Index >= sz )
		{
			Msg(0, "INFO:   '%s' .   ,  gcomp   -vv\n", Props[i].Name);
			continue;
		}

		CMMSObject* Value = Object->GetPropByNum(MMS_Index);
		PropSet.SetValue(this, i, Value->sID);
	}

	ValueType = ToRealType(ValueType);
	//  .      .
	quote_linefeeds(LongHint);
	quote_linefeeds(ShortHint);
}

void CFormColumn::BeautyString(CString& String, int level)
{
	add_parameter(String, level, Class, Identifier, "", true);
	add_offset(String, level);
	String += "{\r\n";

	level++;

	int OffsetClass = (char*)&Class - (char*)this;
	int OffsetIdentifier = (char*)&Identifier - (char*)this;
	int OffsetMask = (char*)&Mask - (char*)this;

	for( int i = 0; i < PropSet.nProps; i++ )
	{
		if( Props[i].Offset == OffsetClass ) continue;
		if( Props[i].Offset == OffsetIdentifier ) continue;

		if( Props[i].Offset == OffsetMask ) //      
		{
			if( ValueType == "" || ValueType == "" || ValueType == "11" )
				add_parameter(String, level, "", Mask, "");
			else
				add_parameter(String, level, "", Mask, "");
		}
		else
			PropSet.BeautyString(String, level, this, i);
	}

	level--;

	add_offset(String, level);
	String += "}\r\n";
}


void CFormControls::FromMMSObject(CMMSObject* mms_controls)
{
	int nControls = mms_controls->GetNProps();
	
	SetSize(nControls);
	for( int i = 0; i < nControls; i++ )
	{
		operator[](i).FromMMSObject( mms_controls->GetPropByNum(i) );
	}
}

void CFormControls::BeautyString(CString& String)
{
	int nControls = GetSize();
	
	if( nControls == 0 )
		return;

	add_offset(String, 1);
	String += "\r\n";
	add_offset(String, 1);
	String += "{\r\n";

	for( int i = 0; i < nControls; i++ )
	{
		operator[](i).BeautyString(String, 2);
	}

	add_offset(String, 1);
	String += "}\r\n";
}

/////////////////////////////////////////////////////////////////
//////  CFormBrowser  ///////////////////////////////////////////
/////////////////////////////////////////////////////////////////
void CFormBrowser::FromMMSObject(CMMSObject* mms_browser)
{
	Param1 = atoi( mms_browser->GetPropByNum(0)->sID );
	Param2 = atoi( mms_browser->GetPropByNum(1)->sID );

	CMMSObject* mms_multicolumn = mms_browser->GetPropByNum(2);
	CMMSObject* mms_fixed = mms_browser->GetPropByNum(3);
	
	Multicolumn.FromMMSObject(mms_multicolumn->GetPropByNum(0));
	
	int nFixed = mms_fixed->GetNProps();
	Fixed.SetSize(nFixed);
	for( int i = 0; i < nFixed; i++ )
	{
		Fixed[i].FromMMSObject(mms_fixed->GetPropByNum(i));
	}
}

void CFormBrowser::BeautyString(CString& String)
{
	int nFixed = Fixed.GetSize();
	
	if( nFixed == 0 )
		return;

	add_offset(String, 1);
	String += "Browser\r\n";
	add_offset(String, 1);
	String += "{\r\n";
	
	add_parameter(String, 2, "Param1", Param1, 0);
	add_parameter(String, 2, "Param2", Param2, 0);

	Multicolumn.BeautyString(String, 2);

	add_offset(String, 2);
	String += "\r\n";
	add_offset(String, 2);
	String += "{\r\n";
	for( int i = 0; i < nFixed; i++ )
	{
		Fixed[i].BeautyString(String, 3);
	}
	add_offset(String, 2);
	String += "}\r\n";

	add_offset(String, 1);
	String += "}\r\n";
}

/////////////////////////////////////////////////////////////////
//////  CDialogForm  ////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
CDialogForm::CDialogForm()
{
	Width = Height = 0;
	Flags1 = 0;
	FontDefault = true;
	AutoTabOrder = true;
	BackgroundColor = -1; //-1 - default.  hex-: 0xBBGGRR
	Flags2 = 0; // Flags1
	ShowPanel = true;
	AllowResize = true;
	Cnt_Ver = "10001";
}

void CDialogForm::FrameFromMMSObject(CMMSObject* mms_frame)
{
	CMMSObject* FrameProps = mms_frame->GetPropByNum(0);
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetLayers = (char*)&Layers - (char*)this;
	int sz = FrameProps->GetNProps();

	for( int i = 0; i < PropSet.nProps; i++ )
	{
		CObjectProperty &prop = Props[i];
		if( prop.Offset == OffsetFont )
			Font.FromMMSObject(FrameProps, prop.MMS_Index);
		else if( prop.Offset == OffsetLayers )
			Layers.FromMMSObject(FrameProps->GetPropByNum(prop.MMS_Index));
		else if( prop.MMS_Index < sz )
			PropSet.SetValue(this, i, FrameProps->GetPropByNum(prop.MMS_Index)->sID);
		else
			Msg(0, "INFO:   '%s' .   ,  gcomp   -vv\n", prop.Name);
	}
}


void CDialogForm::FromMMSObject(CMMSObject* Object)
{
	int nObjects = Object->GetNProps();
	for( int i = 0; i < nObjects; i++ )
	{
		CMMSObject* obj = Object->GetPropByNum(i);
		if( obj->sID == "Frame" )
			FrameFromMMSObject(obj);
		else if( obj->sID == "Browser" )
			Browser.FromMMSObject(obj);
		else if( obj->sID == "Controls" )
			Controls.FromMMSObject(obj);
		else if( obj->sID == "Cnt_Ver" )
			Cnt_Ver = obj->GetPropByNum(0)->sID;
	}
}


void CDialogForm::BeautyString(CString& String)
{
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetLayers = (char*)&Layers - (char*)this;

	String = "\r\n{\r\n";
	for( int i = 0; i < PropSet.nProps; i++ )
	{
		CObjectProperty &prop = Props[i];
		if( prop.Offset == OffsetFont )
			Font.BeautyString(String, 1);
		else if( prop.Offset == OffsetLayers )
			Layers.BeautyString(String);
		else
			PropSet.BeautyString(String, 1, this, i);
	}
	Browser.BeautyString(String);
	Controls.BeautyString(String);
	String += "}";
}


void CDialogForm::WriteToFile(CString FileName)
{
	CString String;
	BeautyString(String);
	CopyStringToFile(String, FileName);
}
