
#include "Declarations.h"
#include "CommonFunctions.h"
#include "DialogForm.h"
#include "DlgFormFlags.h"
#include "Strings.h"


/////////////////////////////////////////////////////////////
void CFormFont::ReadFromFile(CLexer& Lexer)
{
	enum {start, reading, stop} state = start;
	CString Token, Value;
	int i;

	while( Lexer.GetToken(Token) )
	{
		switch( state )
		{
		case start:
			if( Token == '{' ) state = reading;
			break;
		case reading:
			if( Token == '}' ) 
				return;

			i = PropSet.FindByName(Token);
			if( i >= 0 )
			{
				Lexer.GetRest(Value);
				PropSet.SetValue(this, i, Value);
			}
			else
			{
				Msg(0, "ERR:    (%s)\n", Lexer.FilePosInfo());
				Msg(0, "\t%s:%s\n", Token, Value);
				exit(1);
			}
			break;
		}
	}
}

void CFormFont::ToString(CString& String)
{
	if( FontSize > 0 ) ReadRealFontSize(FontSize);

	int OffsetBold = (char*)&FontBold - (char*)this;
	bool AddComma = false;
	for( int MMS_Index = 0; MMS_Index <= PropSet.MaxMMS_Index; MMS_Index++ )
	{
		int i = PropSet.FindByMMSIndex(MMS_Index);
		if( Props[i].Offset == OffsetBold ) //Bold
		{
			if( Param8 == 0 && Param9 == 0 && Param10 == 0 && Param11 == 0 && Param12 == 0 && Param13 == 0 )
				add_quoted(String, "0");
			else if( FontBold )
				add_quoted(String, "700");
			else
				add_quoted(String, "400");
		}
		else
			PropSet.ToString(String, this, i, AddComma);
		AddComma = true;
	}
}

////////////////////////////////////////////////////////////////////////
void CHotKey::ToString(CString& String)
{
	CString str;
	str.Format("{\"\"%i\"\",\"\"%i\"\"}", Modifier, KeyCode);
	add_quoted(String, str);
}

////////////////////////////////////////////////////////////////////////
void CFormControl::ReadFromFile(CLexer& Lexer)
{
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetHotKey = (char*)&HotKey - (char*)this;

	enum {start, reading, stop} state = start;
	CString token, value;
	while( Lexer.GetToken(token) )
	{
		switch( state )
		{
		case start:
			if( token == '{' ) state = reading;
			break;
		case reading:
			if( token == '}' ) return;

			Lexer.GetRest(value);
			if( token.CollateNoCase("") == 0 )
			{
				Mask = value;
				break;
			}
			
			int i = PropSet.FindByName(token);
			if( i >= 0 )
			{
				if( Props[i].Offset == OffsetFont )
				{
					Font.FontName = value;
					Font.ReadFromFile(Lexer);
				}
				else if( Props[i].Offset == OffsetHotKey )
				{
					HotKey.FromString(value, Lexer);
				}
				else
				{
					PropSet.SetValue(this, i, value);
				}
			}
			break;
		}
	}
}

void CFormControl::ToString(CString& String)
{
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetHotKey = (char*)&HotKey - (char*)this;

	ValueType = FromRealType(ValueType, false);
	unquote_linefeeds(LongHint);
	unquote_linefeeds(ShortHint);

	String += ",\r\n{";
	add_quoted(String, Caption, false); //Caption has negative MMS_Index
	for( int MMS_Index = 0; MMS_Index <= PropSet.MaxMMS_Index; MMS_Index++ )
	{
		int i = PropSet.FindByMMSIndex(MMS_Index);
		if( i < 0 ) continue;
		
		if( Props[i].Offset == OffsetFont )
		{
			String += ",";
			Font.ToString(String);
		}
		else if( Props[i].Offset == OffsetHotKey )
			HotKey.ToString(String);
		else
			PropSet.ToString(String, this, i, true);
	}
	String += "}";
}

void CFormColumn::ReadFromFile(CLexer& Lexer)
{
	enum {start, reading, stop} state = start;
	CString token, value;
	while( Lexer.GetToken(token) )
	{
		switch( state )
		{
		case start:
			if( token == '{' ) state = reading;
			break;
		case reading:
			if( token == '}' ) return;
			
			Lexer.GetRest(value);
			if( token.CollateNoCase("") == 0 )
			{
				Mask = value;
				break;
			}
			
			int i = PropSet.FindByName(token);
			if( i >= 0 )
			{
				PropSet.SetValue(this, i, value);
			}
			break;
		}
	}
}

void CFormColumn::ToString(CString& String)
{
	ValueType = FromRealType(ValueType, true);
	unquote_linefeeds(LongHint);
	unquote_linefeeds(ShortHint);

	String += ",\r\n{";
	add_quoted(String, ColumnType, false); //ColumnType has negative MMS_Index
	add_quoted(String, Caption, true); //Caption has negative MMS_Index
	for( int MMS_Index = 0; MMS_Index <= PropSet.MaxMMS_Index; MMS_Index++ )
	{
		int i = PropSet.FindByMMSIndex(MMS_Index);
		if( i < 0 ) continue;
		PropSet.ToString(String, this, i, true);
	}
	String += "}";
}


//////////////////////////////////////////////////////////////////////////////
void CFormBrowser::ReadFromFile(CLexer& Lexer)
{
	enum {start, read_params, wait_brace_fixed, read_fixed, stop} state = start;
	int nFixed = 0;

	CString token, value;
	while( Lexer.GetToken(token) )
	{
		switch( state )
		{
		case start:
			if( token == '{' ) state = read_params;
			break;
		case read_params:
			if( token == '}' ) 
				return;

			Lexer.GetRest(value);
			if( _stricoll(token, "") == 0 )
				state = wait_brace_fixed;
			else if( _stricoll(token, "Param1") == 0 )
				Param1 = atoi(value);
			else if( _stricoll(token, "Param2") == 0 )
				Param2 = atoi(value);
			else if( _stricoll(token, "browse") == 0 )
			{
				Multicolumn.Class = token;
				Multicolumn.Identifier = value;
				Multicolumn.ReadFromFile(Lexer);
			}
			break;
		case wait_brace_fixed:
			if( token == '{' ) state = read_fixed;
			break;
		case read_fixed:
			if( token == '}' )
			{
				state = read_params;
				break;
			}

			Lexer.GetRest(value);
			Fixed.SetSize(nFixed + 1);
			Fixed[nFixed].Class = token;
			Fixed[nFixed].Identifier = value;
			Fixed[nFixed].ReadFromFile(Lexer);
			nFixed++;
			break;
		}
	}
}

void CFormBrowser::ToString(CString& String)
{
	int sz = Fixed.GetSize();
	
	if( sz == 0 ) return;

	String += ",\r\n{\"Browser\"";
	add_quoted(String, Param1);
	add_quoted(String, Param2);
	String += ",\r\n{\"Multicolumn\"";
	Multicolumn.ToString(String);
	String += "}";

	String += ",\r\n{\"Fixed\"";
	for( int i = 0; i < sz; i++ )
		Fixed[i].ToString(String);

	String += "}}";
}

//////////////////////////////////////////////////////////////////////////////
void CFormControls::ReadFromFile(CLexer& Lexer)
{
	enum {start, read_controls, stop} state = start;
	int nControls = 0;

	CString token, value;
	while( Lexer.GetToken(token) )
	{
		switch( state )
		{
		case start:
			if( token == '{' ) state = read_controls;
			break;
		case read_controls:
			if( token == '}' ) return;

			if( !token.IsEmpty() )
			{
				Lexer.GetRest(value);
				SetSize(nControls + 1);
				operator[](nControls).Class = token;
				operator[](nControls).Identifier = value;
				operator[](nControls).ReadFromFile(Lexer);
				nControls++;
			}
			else
			{
				Msg(0, "ERR:      (%s)\n", Lexer.FilePosInfo());
				exit(1);
			}
			break;
		}
	}
}

void CFormControls::ToString(CString& String)
{
	String += ",\r\n{\"Controls\"";

	int sz = GetSize();
	for( int i = 0; i < sz; i++ )
		operator[](i).ToString(String);

	String += "}";
}

////////////////////////////////////////////////////////////////
void CFormLayers::ReadFromFile(CLexer& Lexer)
{
	enum {start, reading, stop} state = start;
	int nLayers = 0;
	CString token, value;
	char* flags;
	while( Lexer.GetToken(token) )
	{
		switch( state )
		{
		case start:
			if( token == '{' ) state = reading;
			break;
		case reading:
			if( token == '}' ) return;

			Lexer.GetRest(value);
			SetSize(nLayers+1);
			operator[](nLayers).Caption = token;
			flags = strtok((char*)(LPCSTR)value, ", \t");
			while( NULL != flags )
			{
				if( _stricoll(flags, "") == 0 )
					operator[](nLayers).IsVisible = true;
				if( _stricoll(flags, "") == 0 )
					ActiveLayer = nLayers;
				flags = strtok(NULL, ", \t");
			}
			nLayers++;
			break;
		}
	}
}

void CFormLayers::ToString(CString& String)
{
	String += ",\r\n{";
	add_quoted(String, ActiveLayer, false);

	int sz = GetSize();
	for( int i = 0; i < sz; i++ )
	{
		CFormLayer& Layer = operator[](i);
		String += ",\r\n{";
		add_quoted(String, Layer.Caption, false);
		add_quoted(String, Layer.IsVisible);
		String += "}";
	}
	String += "}";
}

///////////////////////////////////////////////////////////////////////////
unsigned int read_flags(char* value, char* FlagList[]);
void CDialogForm::ReadFromFile(CString& FileName)
{
	CLexer Lexer(FileName);
	if( !Lexer.IsValid() )
		return;

	enum {start, read_parameters, stop} state = start;
	CString Token, Value;
	while( Lexer.GetToken(Token) )
	{
		switch( state )
		{
		case stop:
			Msg(0, "ERR:      (%s)\n", Lexer.FilePosInfo());
			exit(1);
		case start:
			if( Token == '{' ) state = read_parameters;
			break;
		case read_parameters:
			if( Token == '}' )
			{
				state = stop;
				break;
			}

			if( _stricoll(Token, "") == 0 )
				Layers.ReadFromFile(Lexer);
			else if( _stricoll(Token, "Browser") == 0 )
				Browser.ReadFromFile(Lexer);
			else if( _stricoll(Token, "") == 0 )
				Controls.ReadFromFile(Lexer);
			else if( _stricoll(Token, "") == 0 )
			{
				Lexer.GetRest(Value);
				Font.FontName = Value;
				Font.ReadFromFile(Lexer);
			}
			else
			{
				int i = PropSet.FindByName(Token);
				if( i < 0 )
				{
					Msg(0, "ERR:     (%s)\n", Lexer.FilePosInfo());
					Msg(0, "\t%s\n", Token);
					exit(1);
				}
				Lexer.GetRest(Value);
				PropSet.SetValue(this, i, Value);
			}
			break;
		}
	}
}

//////////////////////////////////////////////////////////////////////////
void CDialogForm::ToString(CString& String)
{
	int OffsetFont = (char*)&Font - (char*)this;
	int OffsetLayers = (char*)&Layers - (char*)this;

	if( Caption.IsEmpty() ) Caption = " ";

	String = "{\"Dialogs\",\r\n{\"Frame\",\r\n{";
	for( int MMS_Index = -1; MMS_Index <= PropSet.MaxMMS_Index; MMS_Index++ )
	{
		int i = PropSet.FindByMMSIndex(MMS_Index);
		if( i >= 0 )
		{
			CObjectProperty &prop = Props[i];
			if( prop.Offset == OffsetFont )
				Font.ToString(String);
			else if( prop.Offset == OffsetLayers )
				Layers.ToString(String);
			else
				PropSet.ToString(String, this, i, true);
		}
	}
	String += "}}";

	Browser.ToString(String);
	Controls.ToString(String);

	String += ",\r\n{\"Cnt_Ver\"";
	add_quoted(String, Cnt_Ver);
	String += "}}";
	String += "  ";
}
