
#include <stdio.h>
#include <io.h>
#include "Task.h"
#include "Storage.h"
#include "CommonFunctions.h"
#include "StreamInfo.h"
#include "MMS.h"

#import "zlibeng.dll" no_namespace

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//- ,    . 
//       
static CMapStringToPtr ExtractedFiles;
static CMapPtrToPtr    ExtractedObjects;

void AddToExtractedList(CString& FullFName, CMMSObject* object)
{
	ExtractedFiles.SetAt((LPCSTR)FullFName, (void*)object);
	if( object != NULL ) ExtractedObjects.SetAt((void*)object, NULL);
}

void CheckPrune(CString& FullFName)
{
	void *ptr;
	if( !ExtractedFiles.Lookup((LPCSTR)FullFName, ptr) )
	{
		remove((LPCSTR)FullFName);
		Msg(1, "DEL: '%s'\n", (LPCSTR)FullFName);
	}
}

bool ObjectExtracted(CMMSObject* object)
{
	void *ptr;
	return (bool)ExtractedObjects.Lookup((void*)object, ptr);
}

static char _StgErrorString[256];
LPCSTR GetStgErrorString(HRESULT err)
{
	switch( err )
	{
		case STG_E_ACCESSDENIED:
			return "Access denied";
		case STG_E_FILENOTFOUND:
			return "The element with the specified name does not exist";
		case STG_E_FILEALREADYEXISTS:
			return "File already exist";
		case STG_E_INVALIDPOINTER:
			return "The pointer specified for the element was invalid";
		case STG_E_INSUFFICIENTMEMORY:
			return "Insufficient memory";
		case STG_E_INVALIDNAME:
			return "Invalid name";
		case STG_E_TOOMANYOPENFILES:
			return "Too many open files";
	}
	sprintf(_StgErrorString, "Unknown error - %l", (long)err);
	return _StgErrorString;
}


////////////////////////////////////////////////////////////////////
CStorage::CStorage()
{
	ReleaseRootStorage = true;
	pStream = NULL;
}

CStorage::CStorage(CString& CompoundName)
{
	IStorage* pRootStorage;
	HRESULT hr;
	wchar_t* wCompoundName = new wchar_t[CompoundName.GetLength()+1];
	
	swprintf(wCompoundName , L"%S", (LPCSTR)CompoundName);
	hr = StgCreateDocfile(wCompoundName, 
		STGM_DIRECT | STGM_READWRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE, 0, &pRootStorage);
	delete[] wCompoundName;
	
	if( hr != S_OK )
	{
		Msg(0, "ERR: can't create storage '%s' - %s\n", (LPCSTR)CompoundName, GetStgErrorString(hr));
		exit(1);
	}

	Storages.AddTail(pRootStorage);
	pStream = NULL;
	ReleaseRootStorage = true;
}


CStorage::CStorage(CStorage& Storage)
{
	ReleaseRootStorage = false;
	IStorage* stor = Storage.GetStorage();
	if( stor != NULL ) Storages.AddTail(stor);
	pStream = NULL;
}

CStorage::CStorage(CStorage& Storage, CString SubStorage)
{
	ReleaseRootStorage = false;
	IStorage* stor = Storage.GetStorage();
	if( stor != NULL ) Storages.AddTail(stor);
	pStream = NULL;
	Open(SubStorage);
}

CStorage::~CStorage()
{
	CloseStream();

	POSITION pos = Storages.GetTailPosition();
	while( (pos = Storages.GetTailPosition()) != NULL )
	{
		IStorage* Storage = Storages.GetAt(pos);
		if( (Storages.GetCount() > 1) || ReleaseRootStorage ) Storage->Release();
		Storages.RemoveTail();
	}
}


IStorage* CStorage::GetStorage()
{
	POSITION pos = Storages.GetTailPosition();
	if( pos != NULL ) 
		return Storages.GetAt(pos);
	else
		return NULL;
}

CString CStorage::FullPath()
{
	CString str;
	POSITION pos = Storages.GetHeadPosition();
	while( pos != NULL )
	{
		STATSTG  stat;
		IStorage* Storage = Storages.GetAt(pos);
		CString stor;
		Storage->Stat(&stat, STATFLAG_DEFAULT);
		stor.Format("\\%S", stat.pwcsName);
		str += stor;
		Storages.GetNext(pos);
	}
	return str;
}

bool CStorage::Open(CStringArray* StorageName)
{
	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	int i, max_i = StorageName->GetSize() - 1;
	for( i = 0; i <= max_i; i++ )
	{
		Parent = OpenStorage(Parent, StorageName->GetAt(i));
		if( Parent == NULL ) return false;
		Storages.AddTail(Parent);
	}

	return true;
}

bool CStorage::Open(CString& StorageName)
{
	if( Storages.GetCount() == 0 )
	{
		IStorage* pRootStorage = OpenCompound(StorageName);
		if( pRootStorage == NULL ) return false;
		Storages.AddTail(pRootStorage);
		ReleaseRootStorage = true;
		return true;
	}

	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	if( StorageName.Find("\\") == -1 )
	{
		IStorage* stor = OpenStorage(Parent, StorageName);
		if( stor == NULL ) return false;
		Storages.AddTail(stor);
	}
	else
	{
		CStringArray* name_parts = SplitPath(StorageName);
		bool res = Open(name_parts);
		delete name_parts;
		return res;
	}

	return true;
}

bool CStorage::Open(const wchar_t* StorageName)
{
	CString sStorageName;
	sStorageName.Format("%S", StorageName);
	return Open(sStorageName);
}

bool CStorage::Create(CStringArray* name_parts)
{
	for( int i = 0; i < name_parts->GetSize(); i++ )
	{
		if( !Create(name_parts->GetAt(i)) ) return false;
	}
	return true;
}

bool CStorage::Create(CString& StorageName)
{
	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	if( StorageName.Find("\\") >= 0 )
	{
		CStringArray* name_parts = SplitPath(StorageName);
		bool res = Create(name_parts);
		delete name_parts;
		return res;
	}

	if( Open(StorageName) ) return true;

	IStorage* pStorage;
	HRESULT hr;
	wchar_t* wStorageName = new wchar_t[StorageName.GetLength()+1];
	swprintf(wStorageName, L"%S", (LPCSTR)StorageName);

	hr = GetStorage()->CreateStorage(wStorageName,
		STGM_DIRECT | STGM_READWRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE, 0,0, 
		&pStorage);
	delete[] wStorageName;

	if( hr != S_OK )
	{
		Msg(0, "ERR: Cant create storage '%s' in '%s' - %s\n", (LPCSTR)StorageName, (LPCSTR)FullPath(), GetStgErrorString(hr));
		return false;
	}

	Storages.AddTail(pStorage);
	return true;
}

bool CStorage::CreateStream(CString& StreamName)
{
	CloseStream();

	if( StreamName.Find("\\") >= 0 )
	{
		CStringArray* name_parts = SplitPath(StreamName);
		this->StreamName = name_parts->GetAt(name_parts->GetSize()-1);

		name_parts->SetSize(name_parts->GetSize()-1);
		bool res = Create(name_parts);
		delete name_parts;
		if( !res ) return false;
	}
	else
		this->StreamName = StreamName;

	HRESULT hr;
	wchar_t* wStreamName = new wchar_t[this->StreamName.GetLength()+1];
	swprintf(wStreamName, L"%S", (LPCSTR)(this->StreamName));


	hr = GetStorage()->CreateStream(wStreamName, 
		STGM_DIRECT | STGM_READWRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE,
		0,0, &pStream);
	delete[] wStreamName;
	
	if( S_OK != hr )
	{
		Msg(0, "ERR: Can't create stream '%s' - %s\n", (LPCSTR)StreamName, GetStgErrorString(hr));
		exit(1);
	}

	StreamInfo.UpdateInfo((char*)(LPCSTR)(this->StreamName));
	return true;
}

bool CStorage::Delete(CString& StorageName)
{
	wchar_t* wStorageName = new wchar_t[StorageName.GetLength()+1];
	HRESULT hr;
	
	swprintf(wStorageName, L"%S", (LPCSTR)StorageName);
	hr = GetStorage()->DestroyElement(wStorageName);
	delete[] wStorageName;

	if( S_OK != hr )
	{
		Msg(0, "ERR: Can't delete element '%s' - %s\n", (LPCSTR)StorageName, GetStgErrorString(hr));
		exit(1);
	}
	return true;
}

void CStorage::Close()
{
	CloseStream();

	POSITION pos = Storages.GetTailPosition();
	if( pos != NULL )
	{
		IStorage* Storage = Storages.GetAt(pos);
		Storage->Release();
		Storages.RemoveTail();
	}
}

void CStorage::CloseStream()
{
	if( pStream != NULL )
	{
		pStream->Release();
		pStream = NULL;
	}
}

bool CStorage::OpenStream(CString& StreamName)
{
	CloseStream();

	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	if( StreamName.Find("\\") == -1 )
	{
		pStream = ::OpenStream(Parent, StreamName);
		this->StreamName = StreamName;
	}
	else
	{
		CStringArray* name_parts = SplitPath(StreamName);
		CString RealName = name_parts->GetAt(name_parts->GetSize()-1);
		name_parts->SetSize(name_parts->GetSize()-1);
		if( !Open(name_parts) ) return false;
		delete name_parts;
		pStream = ::OpenStream(GetStorage(), RealName);
		this->StreamName = RealName;
	}

	if( pStream == NULL ) return false;
	
	StreamInfo.UpdateInfo((char*)(LPCSTR)(this->StreamName));

	return true;
}

bool CStorage::StorageExist(CString& StorageName)
{
	IStorage* pStorage = ::OpenStorage(GetStorage(), StorageName);
	if( pStorage != NULL )
	{
		pStorage->Release();
		return true;
	}
	return false;
}

bool CStorage::StreamExist(CString& StreamName)
{
	IStream* pStream = ::OpenStream(GetStorage(), StreamName);
	if( pStream != NULL )
	{
		pStream->Release();
		return true;
	}
	return false;
}

long CStorage::GetStreamSize(CString& StreamName)
{
	long size;
	if( !OpenStream(StreamName) ) return -1L;
	size = ::GetStreamSize(pStream).LowPart;
	CloseStream();
	return size;
}

void CStorage::Rewind()
{
	LARGE_INTEGER SeekZero = {0, 0};
	pStream->Seek(SeekZero, 0, NULL);
}

void CStorage::GetCleanStream()
{
	if( StreamInfo.Packed )
	{
		IStream* pUnpackedStream;
		CreateStreamOnHGlobal(NULL, TRUE, &pUnpackedStream);

		IzlibEnginePtr iLibEngine;
		iLibEngine.CreateInstance(L"V75.zlibEngine");
		iLibEngine->pkDecompress(pStream, pUnpackedStream);

		pStream->Release();
		pStream = pUnpackedStream;
	}

	if( StreamInfo.HaveSize )
	{
		StreamInfo.ReadSize(pStream); //   ,      
		
		LARGE_INTEGER Offset = {0, 0};
		Offset.LowPart = StreamInfo.ContentOffset;
		pStream->Seek(Offset, STREAM_SEEK_SET, NULL);
		
		IStream* CleanStream;
		CreateStreamOnHGlobal(NULL, TRUE, &CleanStream);
		
		ULARGE_INTEGER BytesToCopy = {0, 0};
		BytesToCopy.LowPart = StreamInfo.FullSize - StreamInfo.SizeBytes - StreamInfo.SizeOffset;
		pStream->CopyTo(CleanStream, BytesToCopy, NULL, NULL);

		Offset.LowPart = 0;
		CleanStream->Seek(Offset, 0, NULL);

		pStream->Release();
		pStream = CleanStream;
	}
	else
	{
		StreamInfo.ReadSize(pStream);
	}

}

bool CStorage::CopyToFile(FILE* File, unsigned long bytes_to_copy)
{
	if( File == NULL ) return false;

	BYTE buff[BUFFER_SIZE];
	unsigned long BytesToRead = BUFFER_SIZE, BytesRead = 0;
	DWORD BytesWritten = 0;

	if( bytes_to_copy <= 0 ) Rewind();

	do
	{
		if( bytes_to_copy > 0 && BytesToRead > bytes_to_copy )
			BytesToRead = bytes_to_copy;

		pStream->Read(buff, BytesToRead, &BytesRead);

		if( BytesRead > 0 )
		{
			BytesWritten = fwrite(buff, 1, BytesRead, File);
			if( BytesWritten < BytesRead ) return false;
		}
		
		if( bytes_to_copy > 0 )
		{
			if( bytes_to_copy <= BytesRead ) break;
			bytes_to_copy -= BytesRead;
		}

	} while( BUFFER_SIZE == BytesRead );
	return true;
}

bool CStorage::CopyToFile(CString& FileName, bool OnlyRestOfSream)
{
	if( pStream == NULL ) return false;

	FILE *File;
	char* MsgPrefix = "";
	unsigned long bytes_to_copy = 0;

	//  
	if( StreamInfo.Picture )
	{
		StreamInfo.DeterminePictureType(pStream);
		FileName += StreamInfo.PictureType;
	}
	else if( StreamInfo.PictureGallery )
	{
		return ExtractPictureGallery(FileName);
	}

	if( OnlyRestOfSream )
	{
		LARGE_INTEGER Offset = {0, 0};
		ULARGE_INTEGER Pos = {0, 0};
		pStream->Seek(Offset, STREAM_SEEK_CUR, &Pos);
		bytes_to_copy = StreamInfo.FullSize - Pos.LowPart;
		if( bytes_to_copy == 0 ) return true;
	}

	AddToExtractedList(FileName);

	if( FileExist(FileName) )
	{
		if( bytes_to_copy == 0 )
		{
			if( FullCompare(FileName) == 0 ) return true;
		}
		else if( Compare(FileName, bytes_to_copy) == 0 )
		{
			return true;
		}
		MsgPrefix = "UPD";
	}
	else
	{
		MsgPrefix = "NEW";
	}

	File = fopen(FileName, "wb");
	if( !CopyToFile(File, bytes_to_copy) )
	{
		Msg(0, "Error writing to file '%s' - %s\n", FileName, strerror(errno));
		exit(1);
	}

	fclose(File);

	Msg(1, "%s: %s\n", MsgPrefix, FileName);
	return true;
}

bool CStorage::CopyToFile(CString& StreamName, CString& FileName)
{
	if( !OpenStream(StreamName) ) return false;

	GetCleanStream();

	if( StreamInfo.PictureGallery )
	{
		return ExtractPictureGallery(FileName);
	}

	if( StreamInfo.HaveSize )
	{
		if( (StreamInfo.Size != (StreamInfo.FullSize - StreamInfo.SizeBytes - StreamInfo.SizeOffset)) )
			Msg(0, "WARNING: Checksum %ib != real size %ib. Stream '%s\\%s'. File '%s'\n", 
				(int)StreamInfo.Size, 
				(int)(StreamInfo.FullSize - StreamInfo.SizeBytes - StreamInfo.SizeOffset),
				FullPath(), StreamName,
				FileName
				);
	}

	return CopyToFile(FileName);
}

bool CStorage::Extract(CString& DestDir, bool WithContainerContents)
{
	IEnumSTATSTG *ppenum;
	STATSTG stat;
	unsigned long uCount;
	CString StreamName, FileName, cc("Container.Contents");

	CreateDirectory(DestDir);

	GetStorage()->EnumElements(0, NULL, 0, &ppenum);
	while ( S_OK == ppenum->Next(1, &stat, &uCount) )
	{
		StreamName.Format("%S", stat.pwcsName);
		FileName.Format("%s\\%s", DestDir, StreamName);

		if( (stat.type & STGTY_STORAGE) != 0 )
		{
			if( Open(StreamName) )
			{
				Extract(FileName, WithContainerContents);
				Close();
			}
		}
		else
		{
			if( !WithContainerContents && StreamName == cc ) continue;
			CopyToFile(StreamName, FileName);
		}
	}
	ppenum->Release();

	return true;
}

unsigned long CStorage::AppendFile(IStream* pStream, FILE* File)
{
	char buff[BUFFER_SIZE];
	size_t BytesRead;
	unsigned long TotalBytesRead = 0;

	LARGE_INTEGER offset;
	ULARGE_INTEGER pos = ::GetStreamSize(pStream);
	offset.QuadPart = pos.QuadPart;
	pStream->Seek(offset, STREAM_SEEK_SET, &pos);

	do
	{
		BytesRead = fread(buff, 1, sizeof(buff), File);
		pStream->Write(buff, BytesRead, NULL);
		TotalBytesRead += BytesRead;
	} while( BytesRead == sizeof(buff) );

	return TotalBytesRead;
}

unsigned long CStorage::AppendFile(IStream* pStream, CString& FileName)
{
	FILE* File = fopen(FileName, "rb");
	if( File == NULL ) return 0;

	unsigned long bytes_appended = AppendFile(pStream, File);
	fclose(File);
	return bytes_appended;
}

bool CStorage::FromFile(FILE* File, LPCSTR FileName)
{
	IStream* pStreamText;

	if( StreamInfo.Packed )
		CreateStreamOnHGlobal(NULL, TRUE, &pStreamText);
	else
		pStreamText = pStream;

	if( StreamInfo.HaveSize )
	{
		StreamInfo.ReadFileSize(File); //,   0xFF   
		WriteSizeToStream(pStreamText);
	}

	rewind(File);
	AppendFile(pStreamText, File);

	if( StreamInfo.Packed )
	{
		LARGE_INTEGER SeekZero = {0, 0};
		pStreamText->Seek(SeekZero, STREAM_SEEK_SET, NULL);

		IzlibEnginePtr iLibEngine;
		iLibEngine.CreateInstance(L"V75.zlibEngine");
		iLibEngine->pkCompress(pStreamText, pStream);
		
		pStreamText->Release();
	}

	if( FileName != NULL ) Msg(2, "CPY: '%s'\n", FileName);

	return true;
}

bool CStorage::FromFile(CString& FileName)
{
	FILE* File;
	File = fopen((LPCSTR)FileName, "rb");
	if( File == NULL ) return false;

	FromFile(File, FileName);
	fclose(File);
	return true;
}

bool CStorage::StreamFromFile(CString& StreamName, CString& FileName)
{
	if( !FileExist(FileName) ) return false;

	StreamInfo.UpdateInfo(StreamName);
	if( StreamInfo.Picture )
	{
		StreamInfo.DeterminePictureType(FileName);
		if( !StreamInfo.PictureType.IsEmpty() )
		{
			int offset = StreamName.GetLength() - StreamInfo.PictureType.GetLength();
			StreamName.SetAt(offset, '\0'); // 
		}
	}
	else if( StreamInfo.PictureGallery )
	{
		return ImportPictureGallery(StreamName, FileName);
	}

	CString PictureType = StreamInfo.PictureType; // 
	if( !CreateStream(StreamName) ) return false;
	StreamInfo.PictureType = PictureType;

	return FromFile(FileName);
}

void CStorage::AddZeroByte()
{
	BYTE byte = 0;
	if( pStream == NULL ) return;
	pStream->Write(&byte, 1, NULL);
}

bool CStorage::StreamFromString(CString StreamName, CString& String)
{
	if( !CreateStream(StreamName) ) return false;

	StreamInfo.ReadSize(String);
	
	WriteSizeToStream(pStream);
	pStream->Write((LPCSTR)String, String.GetLength(), NULL);
	return true;
}

bool CStorage::StorageFromDir(CString& StorageName, CString& SrcDir)
{
	CString FileName;
	struct _finddata_t find_data;
	long hFind;

	if( !DirectoryExist(SrcDir) ) return false;

	if( !Create(StorageName) )
	{
		Msg(0, "ERR: Can't create storage %s\n", StorageName);
		return false;
	}

	GalleryImported = false;
	hFind = _findfirst(SrcDir+"\\*", &find_data);
	if( hFind != -1 )
	{
		do
		{
			if( IsIgnoredFile(find_data.name) ) continue;
			
			FileName.Format("%s\\%s", SrcDir, find_data.name);

			if( (find_data.attrib & _A_SUBDIR) != 0 )
			{
				StorageFromDir(CString(find_data.name), FileName);
				Close();
			}
			else
				StreamFromFile(CString(find_data.name), FileName);
		} while( _findnext(hFind, &find_data) == 0 );
		_findclose(hFind);
	}

	return true;
}

void CStorage::WriteSizeToStream(IStream* pStream)
{
	BYTE byte_ff = 0xFF;

	if( !StreamInfo.HaveSize ) return;

	if( StreamInfo.Picture )
	{
		//   -    As Is.    
		if( StreamInfo.PictureType.IsEmpty() ) return;

		unsigned char signature[] = {0x6c, 0x74, 0x00, 0x00};
		pStream->Write(signature, sizeof(signature), NULL);
	}
	else
	{
		//  0xFF
		for( unsigned int i = 0; i < StreamInfo.SizeOffset; i++ ) 
			pStream->Write(&byte_ff, 1, NULL);
	}


	switch( StreamInfo.SizeBytes )
	{
		BYTE bSize;
		WORD wSize;
		DWORD dwSize;
		case 1:
			bSize = (BYTE)StreamInfo.Size;
			pStream->Write(&bSize, StreamInfo.SizeBytes, NULL);
			break;
		case 2:
			wSize = (WORD)StreamInfo.Size;
			pStream->Write(&wSize, StreamInfo.SizeBytes, NULL);
			break;
		case 4:
			dwSize = StreamInfo.Size;
			pStream->Write(&dwSize, StreamInfo.SizeBytes, NULL);
			break;
		default:
			Msg(0, "ERR: Don't know how to write size of stream (%i bytes)\n", (int)StreamInfo.SizeBytes);
			exit(1);
			break;
	}
}


//   ,  -1
//   ,  1
//  ,    memcmp()
int CStorage::Compare(FILE* File, unsigned long bytes_to_compare)
{
	//save stream position
	LARGE_INTEGER seek_offset = {0, 0};
	ULARGE_INTEGER pos;
	pStream->Seek(seek_offset, STREAM_SEEK_CUR, &pos);

	void* buff_Stream[BUFFER_SIZE];
	void* buff_File[BUFFER_SIZE];
	DWORD BytesReadF = 0, BytesReadS = 0;
	int cmp = 0;

	do
	{
		BytesReadF = fread(buff_File, 1, BUFFER_SIZE, File);
		pStream->Read(buff_Stream, BUFFER_SIZE, &BytesReadS);

		if( bytes_to_compare > 0 )
		{
			if( BytesReadF > bytes_to_compare ) BytesReadF = bytes_to_compare;
			if( BytesReadS > bytes_to_compare ) BytesReadS = bytes_to_compare;
		}
		
		if( BytesReadF < BytesReadS ) 
		{
			cmp = -1;
			break;
		}
		if( BytesReadF > BytesReadS ) 
		{
			cmp = 1;
			break;
		}

		cmp = memcmp(buff_File, buff_Stream, BytesReadF);
		if( cmp != 0 ) break;

		if( bytes_to_compare > 0 )
		{
			if( bytes_to_compare <= BytesReadF ) break;
			bytes_to_compare -= BytesReadF;
		}
	} while( BUFFER_SIZE == BytesReadF );
	
	//restore stream position
	seek_offset.QuadPart = pos.QuadPart;
	pStream->Seek(seek_offset, STREAM_SEEK_SET, NULL);

	return cmp;
}

int CStorage::Compare(CString& FileName, unsigned long bytes_to_compare)
{
	if( !FileExist(FileName) ) return true;

	FILE* File = fopen(FileName, "rb");
	if (File == NULL ) 
	{ 
		Msg(0, "ERR: Can\t open file '%s' - %s\n", FileName, strerror(errno));
		exit(1);
	}

	int cmp = Compare(File, bytes_to_compare);

	fclose(File);

	return cmp;
}

int CStorage::FullCompare(CString& FileName)
{
	if( !FileExist(FileName) ) return 0;
	long FileSize = GetFileSize(FileName);
	int cmp = -1;
	
	if ( FileSize != StreamInfo.Size )
	{
		//  
		Msg(2, "File size is: %d - %s\n", FileSize, FileName);
		Msg(2, "Strm size is: %d\n", StreamInfo.Size);
		cmp = (FileSize < StreamInfo.Size ? -1 : 1);
	}
	else
	{
		Rewind();
		cmp = Compare(FileName, 0);
	}

	return cmp;
}

extern int MMS_debug;
CMMSObject* CStorage::ParseStream(CString& StreamName)
{
	FILE* File;
	CMMSObject* MMSObject;

	if( !OpenStream(StreamName) ) return NULL;
	GetCleanStream();

	File = tmpfile();
	if( File == NULL )
	{
		Msg(0, "ERR: Can't create temporary file\n");
		exit(1);
	}

	CString FullStreamPath = FullPath()+"\\"+StreamName;

	if( !CopyToFile(File, 0) )
	{
		Msg(0, "ERR: Error extracting '%s' into temporary file\n", FullStreamPath);
		exit(1);
	}
	rewind(File);

	MMS_InitParser(FullStreamPath);
	if( TaskParameters.Verbose >= 3 )
	{
		MMS_debug = 1;
		fprintf(stderr, "===== Start parse '%s' ===\n", FullStreamPath);
	}
	int RetCode = MMS_parse(File, &MMSObject);
	fclose(File);
	if( RetCode != 0 ) 
	{
		Msg(0, "ERR: Error parsing '%s': stream is corrupted, or password protected.\n", FullStreamPath);
		return NULL;
	}

	return MMSObject;
}

////////////////////////////////////////////////////////////////////////////////////////
ULARGE_INTEGER GetStreamSize(IStream* pStream)
{
	STATSTG stat;
	pStream->Stat(&stat, STATFLAG_NONAME);
	return stat.cbSize;
}

IStorage* OpenCompound(CString& CompoundFileName)
{
	IStorage* pRootStorage;
	wchar_t* wCompoundFileName = new wchar_t[CompoundFileName.GetLength()+1];

	swprintf(wCompoundFileName, L"%S", CompoundFileName);
	HRESULT hr = StgOpenStorage(wCompoundFileName , NULL, 
		STGM_SHARE_EXCLUSIVE | STGM_DIRECT | STGM_READWRITE
		, NULL, 0, 
		&pRootStorage);

	delete[] wCompoundFileName;

	if( S_OK != hr )
	{
		Msg(0, "ERR: Can not open storage '%s' - %s\n", 
			(LPCSTR)CompoundFileName,  GetStgErrorString(hr));
		exit(1);
	}

	return pRootStorage;
}

IStorage* OpenStorage(IStorage* pParentStorage, wchar_t* wStorName)
{
	IStorage* pStorage;
	pParentStorage->OpenStorage(wStorName, NULL, STGM_SHARE_EXCLUSIVE | STGM_READWRITE, NULL, 0, &pStorage);
	return pStorage;
}

IStorage* OpenStorage(IStorage* pParentStorage, CString& StorName)
{
	IStorage* pStorage;

	wchar_t* wStorName = new wchar_t[StorName.GetLength()+1];
	swprintf(wStorName, L"%S", StorName);
	pStorage = OpenStorage(pParentStorage, wStorName);
	delete[] wStorName;

	return pStorage;
}

IStream* OpenStream(IStorage* pParentStorage, wchar_t* wStreamName)
{
	IStream* pStream;
	pParentStorage->OpenStream(wStreamName, NULL, STGM_SHARE_EXCLUSIVE, 0, &pStream);
	return pStream;
}

IStream* OpenStream(IStorage* pParentStorage, CString& StreamName)
{
	IStream* pStream;
	
	wchar_t* wStreamName = new wchar_t[StreamName.GetLength()+1];
	swprintf(wStreamName, L"%S", StreamName);
	pStream = OpenStream(pParentStorage, wStreamName);
	delete[] wStreamName;

	return pStream;
}


bool StringDifferFromFile(CString& String, CString& FileName)
{

	if( !FileExist(FileName) ) return true;

	FILE* File = fopen(FileName, "rb");
	if (File == NULL ) 
	{ 
		Msg(0, "ERR: Can\t open file '%s' - %s\n", FileName, strerror(errno));
		exit(1);
    }

	long FileSize;
	long BytesRead;
	void* buff = NULL;
	
	fseek(File, 0, SEEK_END);
	FileSize = ftell(File);
	rewind(File);

	if ( String.GetLength() != FileSize ) goto FilesDiffer;

	buff = (void*) new char[FileSize];
	BytesRead = fread(buff, 1, FileSize, File);
	if( memcmp(buff, (LPCSTR)String, FileSize) != 0 ) goto FilesDiffer;

	delete[] buff;
	fclose(File);
	return false;

FilesDiffer:
	if( buff != NULL ) delete[] buff;
	fclose(File);
	return true;
}

bool CopyStringToFile(CString& String, CString& FileName)
{
	char* MsgPrefix;

	AddToExtractedList(FileName);

	if( FileExist(FileName) )
		if( !StringDifferFromFile(String, FileName) )
			return true;
		else
			MsgPrefix = "UPD";
	else
		MsgPrefix = "NEW";

	//   .        
	//   .     .
	FILE *File = fopen(FileName, "wb");

	fwrite((LPCSTR)String, 1, String.GetLength(), File);
	fclose(File);
	
	Msg(1, "%s: %s\n", MsgPrefix, FileName);

	return true;
}



