
#include "UI.h"
#include "CommonFunctions.h"
#include "NameTable.h"
#include "Storage.h"
#include "MMS.h"
#include "Strings.h"


bool ReadUIString(CString& String, IStream* pStream)
{
	BYTE len;
	char buf[256];
	ULONG BytesRead;

	pStream->Read(&len, sizeof(len), &BytesRead);
	if( BytesRead != sizeof(len) ) return false;
	pStream->Read(buf, len, &BytesRead);
	if( BytesRead != len ) return false;

	buf[len] = '\0';
	String = buf;
	return true;
}



CString CUICommand::AsString(bool IsChild)
{
	CString res;

	if( IsChild )
	{
		res.Format("\t%s: %s\r\n", Command, CmdParameter);
	}
	else
	{
		CString strHotKey;
		if( HotKey != 0 ) strHotKey.Format("%02x:%04x", HotKeyModifier, HotKey);
		res.Format("\t%08x:\r\n\t{\r\n", ID);
		add_parameter(res, 2, "1", Param1, NULL);
		add_parameter(res, 2, "Hint", Hint, NULL);
		add_parameter(res, 2, "HotKey", strHotKey, NULL);
		add_parameter(res, 2, "", Command, NULL);
		add_parameter(res, 2, "", CmdParameter, NULL);
		res += "\t}\r\n";
	}
	return res;
}


CUICommands::CUICommands()
{
	ID = 0x08;
	NCommands = 0;
	IsChild = false;
	NoOperMenuPresent = 0;
	IsDefault = 0;
}



DWORD CUICommands::Read(CStorage& Storage, CString& UIName)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UICommands);
	Storage.OpenStream(nt_rec->StorageName);
	IStream* pStream = Storage.pStream;

	DWORD i;
	
	InterfaceName = UIName;

	pStream->Read(&ID, sizeof(ID), NULL);
	if( IsChild )
	{
		ReadUIString(ParentInterface, pStream);
	}
	else
	{
		ReadUIString(Description, pStream);
	}

	pStream->Read(&NoOperMenuPresent, sizeof(NoOperMenuPresent), NULL);
	pStream->Read(&IsDefault, sizeof(IsDefault), NULL);

	if( IsChild )
	{
		ReadUIString(Description, pStream);
	}
	
	pStream->Read(&NCommands, sizeof(NCommands), NULL);

	Commands.SetSize(NCommands);
	for( i = 0; i < NCommands; i++ )
	{
		CUICommand& cmd = Commands[i];
		
		if( IsChild )
		{
			if( !ReadUIString(cmd.Command, pStream) )
			{
				Msg(0, "WARN:   %s  %i ,    %i\n", UIName, NCommands, i);
				NCommands = i;
				break;
			}
			if( !ReadUIString(cmd.CmdParameter, pStream) )
			{
				Msg(0, "WARN:   %s  %i ,    %i\n", UIName, NCommands, i);
				NCommands = i;
				break;
			}
		}
		else
		{
			pStream->Read(&cmd.ID, sizeof(cmd.ID), NULL);
			
			ReadUIString(cmd.Param1, pStream);
			ReadUIString(cmd.Hint, pStream);
			
			pStream->Read(&cmd.HotKeyModifier, sizeof(cmd.HotKeyModifier), NULL);
			pStream->Read(&cmd.HotKey, sizeof(cmd.HotKey), NULL);

			ReadUIString(cmd.Command, pStream);
			ReadUIString(cmd.CmdParameter, pStream);
		}
		cmd.CmdParameter.Replace('\n', '~~');
	}

	return NCommands;
}

CString CUICommands::AsString()
{
	CString res;

	add_parameter(res, 0, "", StorageName, NULL);
	add_parameter(res, 0, "", IsChild ? "" : "", NULL);
	add_parameter(res, 0, "", InterfaceName, NULL);
	add_parameter(res, 0, "", ParentInterface, NULL);
	add_parameter(res, 0, "", Description, NULL);
	add_parameter(res, 0, "", NoOperMenuPresent == 0 ? "" : "", "");
	add_parameter(res, 0, "", IsDefault == 0 ? "" : "", "");

	res += "{\r\n";
	for( DWORD i = 0; i < NCommands; i++ )
	{
		res += Commands[i].AsString(IsChild);
	}
	res += "}";

	return res;
}


CUIMenu::CUIMenu()
{
	Type = MainMenu;
	Parent = NULL;
	Level = 0;
	NItems = 0;
	ID = 0;
}


DWORD CUIMenu::Read(CStorage& Storage, CUIMenu* Parent)
{
	this->Parent = Parent;
	if( Parent != NULL )
	{
		Level = Parent->Level + 1;
	}
	else
	{
		CNameTableRecord* nt_rec = NameTable.Find(NT_UIMenu);
		if( !Storage.OpenStream(nt_rec->StorageName) ) return 0;
	}

	IStream* pStream = Storage.pStream;

	pStream->Read(&ID, sizeof(ID), NULL);

	switch( ID )
	{
		case 0x01      : Type = MainMenu; break;
		case 0xFFFFFFFF: Type = MenuGroup; break;
		case 0x00      : Type = MenuDividor; break;
		default        : Type = MenuItem;
	}

	if( Type == MenuGroup || Type == MenuItem )
	{
		ReadUIString(Caption, pStream);
	}
	else if( Type == MenuDividor )
	{
		Caption = ">----------<";
	}

	if( Type == MainMenu || Type == MenuGroup )
	{
		pStream->Read(&NItems, sizeof(NItems), NULL);
		if( NItems > 0 )
		{
			Items.SetSize(NItems);
			for( DWORD i = 0; i < NItems; i++ )
			{
				Items[i].Read(Storage, this);
			}
		}
		return NItems;
	}

	return 1;
}

CString CUIMenu::AsString()
{
	CString prefix, sID, res;
	
	sID.Format("%08x", ID);
	str_repeate(prefix, "\t", Level);

	res += prefix;
	switch( Type )
	{
		case MainMenu   : res += ":";      break;
		case MenuGroup  : res += ": ";          break;
		case MenuItem   : res += ": "+sID+": "; break;
		case MenuDividor: res += ": ";      break;
	}
	res += Caption;
	res += "\r\n";

	if( Type == MainMenu || Type == MenuGroup )
	{
		res += prefix;
		res += "{\r\n";
		for( DWORD i = 0; i < NItems; i++ )
		{
			res += Items[i].AsString();
		}
		res += prefix;
		res += "}\r\n";
	}

	return res;
}

DWORD CUIPanel::Read(CStorage& Storage)
{
	IStream* pStream = Storage.pStream;
	ReadUIString(Caption, pStream);
	pStream->Read(&NButtons, sizeof(NButtons), NULL);
	pStream->Read(&Placement, sizeof(Placement), NULL);
	pStream->Read(&IsVisible, sizeof(IsVisible), NULL);
	pStream->Read(&OnNewLine, sizeof(OnNewLine), NULL);
	return NButtons;
}

CString CUIPanel::AsString()
{
	CString res;
	CString sPlacement, sVisible, sOnNewLine;

	switch( Placement )
	{
		case 0xE81B: sPlacement = ""; break;
		case 0xE81C: sPlacement = ""; break;
		case 0xE81D: sPlacement = ""; break;
		case 0xE81E: sPlacement = ""; break;
		default: 
			sPlacement = "   !!!";
			Msg(0, "ERR:     %s\n", Caption);
	}
	
	if( IsVisible == 0 )
		sVisible = "";
	else
		sVisible = "";

	if( OnNewLine == 0 )
		sOnNewLine = "";
	else
		sOnNewLine = "";

	res.Format(": %s\r\n{\r\n\t: %s\r\n\t: %s\r\n\t: %s\r\n\t: %i\r\n}\r\n", 
		Caption, sPlacement, sVisible, sOnNewLine, (int)NButtons);

	return res;
}

CUIPanels::CUIPanels()
{
	ID = 0xFFFFFFFD;
	NPanels = 0;
	NButtons = 0;
	NTextButtons = 0;
	NPictureButtons = 0;
}

bool CUIPanels::Read(CStorage& Storage)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UIPanels);
	CString StreamName = nt_rec->PrepareStorageName();
	long sz = Storage.GetStreamSize(StreamName);
	if( !Storage.OpenStream(StreamName) ) return false;
	Storage.StreamInfo.FullSize = sz;

	IStream* pStream = Storage.pStream;
	DWORD i;

	pStream->Read(&ID, sizeof(ID), NULL);
	pStream->Read(&NPanels, sizeof(NPanels), NULL);
	
	Panels.SetSize(NPanels);
	for( i = 0; i < NPanels; i++ )
	{
		NButtons += Panels[i].Read(Storage);
	}

	AllButtons.SetSize(NButtons);
	for( i = 0; i < NButtons; i++ )
	{
		pStream->Read(&AllButtons[i].ID, sizeof(AllButtons[i].ID), NULL);
	}

	pStream->Read(&NTextButtons, sizeof(NTextButtons), NULL);
	TextButtons.SetSize(NTextButtons);
	for( i = 0; i < NTextButtons; i++ )
	{
		ReadUIString(TextButtons[i].Caption, pStream);
		pStream->Read(&TextButtons[i].ID, sizeof(TextButtons[i].ID), NULL);
	}

	pStream->Read(&NPictureButtons, sizeof(NPictureButtons), NULL);
	PictureButtons.SetSize(NPictureButtons);
	for( i = 0; i < NPictureButtons; i++ )
	{
		pStream->Read(&PictureButtons[i].ID, sizeof(PictureButtons[i].ID), NULL);
	}

	LARGE_INTEGER SeekZero = {0, 0};
	ULARGE_INTEGER pos;
	pStream->Seek(SeekZero, STREAM_SEEK_CUR, &pos);
	BMP_offset = pos.LowPart;

	return true;
}

void CUIPanels::ExtractBMP(CString& Dir, CStorage& Storage)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UIPanelsBMP);
	LARGE_INTEGER offset = {BMP_offset, 0};
	CString StreamName = nt_rec->StorageName;
	if( !Storage.OpenStream(StreamName) ) return;
	Storage.pStream->Seek(offset, STREAM_SEEK_SET, NULL);
	Storage.CopyToFile(nt_rec->PrepareFileName(Dir), true);
}

CString CButton::AsString(bool WithCaption)
{
	CString res;
	if( WithCaption )
	{
		res.Format("\t%08x: %s\r\n", ID, Caption);
	}
	else
	{
		res.Format("\t%08x\r\n", ID);
	}

	return res;
}

CString CUIPanels::AsString()
{
	CString res;
	DWORD i;

	for( i = 0; i < NPanels; i++ )
	{
		res += Panels[i].AsString();
	}

	res += ":\r\n{\r\n";
	for( i = 0; i < NButtons; i++ )
	{
		res += AllButtons[i].AsString(false);
	}
	res += "}\r\n";

	res += ":\r\n{\r\n";
	for( i = 0; i < NTextButtons; i++ )
	{
		res += TextButtons[i].AsString(true);
	}
	res += "}\r\n";

	res += ":\r\n{\r\n";
	for( i = 0; i < NPictureButtons; i++ )
	{
		res += PictureButtons[i].AsString(false);
	}
	res += "}\r\n";

	return res;
}

CUserInterfaces::~CUserInterfaces()
{
	int i = GetSize();
	while( --i >= 0 )
	{
		delete GetAt(i);
	}
}

CUserInterface* CUserInterfaces::FindInterface(CString& Name)
{
	int i = GetSize() - 1;
	while( i >= 0 )
	{
		if( GetAt(i)->Name == Name ) return GetAt(i);
		CUserInterface* Interface = GetAt(i)->Children.FindInterface(Name);
		if( Interface != NULL ) return Interface;
		i--;
	}
	return NULL;
}

void CUserInterfaces::AddInterface(CUserInterface* Interface)
{
	if( Interface->Commands.IsChild )
	{
		CUserInterface* Parent = FindInterface(Interface->Commands.ParentInterface);
		Parent->Children.Add(Interface);
		Interface->Parent = Parent;
	}
	else
	{
		Add(Interface);
	}
}

void CUserInterfaces::Extract(CString& Dir, CStorage& Storage)
{
	for( int i = 0; i < GetSize(); i++ )
	{
		GetAt(i)->Extract(Dir, Storage);
	}
}

void CUserInterface::Read(CString& Dir, CStorage& Storage, CMMSObject* Container)
{
	StorageName = Container->GetPropByNum(0)->sID;
	Name = Container->GetPropByNum(1)->sID;
	Msg(2, " %s\n", Name);

	Storage.Open(StorageName);
	
	if( Container->sID == "UsersInterfaceType" )
	{
		Commands.IsChild = false;
	}
	else if( Container->sID == "SubUsersInterfaceType" )
	{
		Commands.IsChild = true;
	}

	Commands.StorageName = StorageName;
	Commands.Read(Storage, Name);
	if( !Commands.IsChild )
	{
		Menu.Read(Storage, NULL);
		Panels.Read(Storage);
	}

	Storage.Close();
}

void CUserInterface::Extract(CString& Dir, CStorage& Storage)
{
	CString UIDir = Dir + "\\" + Translit(Name); 
	// TODO    ,    SbCnts <-> SbCnt. 
	//   Decompile(),   Extract(),  sID,   Name.
	CreateDirectory(UIDir);

	CNameTableRecord* nt_rec = NameTable.Find(NT_UICommands);
	CopyStringToFile(Commands.AsString(), nt_rec->PrepareFileName(UIDir));
	if( !Commands.IsChild )
	{
		nt_rec = NameTable.Find(NT_UIMenu);
		CopyStringToFile(Menu.AsString(), nt_rec->PrepareFileName(UIDir));
		nt_rec = NameTable.Find(NT_UIPanels);
		CopyStringToFile(Panels.AsString(), nt_rec->PrepareFileName(UIDir));

		Storage.Open(StorageName);
		Panels.ExtractBMP(UIDir, Storage);
		Storage.Close();
	}

	Children.Extract(UIDir, Storage);
}

bool DecompileUI(CString& Dir, CStorage& Storage)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UserInterface);
	CString DestDir = nt_rec->PrepareDirName(Dir);
	CStorage UIStorage(Storage);
	CMMSNodeList* CC;
	CUserInterfaces UserInterfaces;

	if( !UIStorage.Open(nt_rec->PrepareStorageName()) )
	{
		return false;
	}

	CC = (CMMSNodeList*)UIStorage.ParseStream(CString("Container.Contents"));
	if( CC == NULL ) return false;

	CreateDirectory(DestDir);

	POSITION pos = CC->GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* UI = CC->GetNext(pos);
		CUserInterface* Interface = new CUserInterface;
		Interface->Read(DestDir, UIStorage, UI);
		UserInterfaces.AddInterface(Interface);
	}

	delete CC;

	UserInterfaces.Extract(DestDir, UIStorage);
	
	return true;
}