
#include "MMS.h"
#include "CommonFunctions.h"


////////////////////////////////////////////////////////////////////////////
void str_repeate(CString& dest, CString src, int count)
{
	while( count > 0 )
	{
		dest += src;
		count--;
	}
}

void add_quoted(CString& dest, CString& value)
{
	dest += "\"";
	dest += value;
	dest += "\"";
}

void add_quoted(CString& dest, const char* value)
{
	add_quoted(dest, CString(value));
}

int PropsCompareByID( const void *arg1, const void *arg2 )
{
	CMMSObject* obj1 = *(PCMMSObject*)arg1;
	CMMSObject* obj2 = *(PCMMSObject*)arg2;
	if( obj1->ID < obj2->ID ) return -1;
	if( obj1->ID > obj2->ID ) return 1;
	return 0;
}

int PropsCompareByName( const void *arg1, const void *arg2 )
{
	CMMSObject* obj1 = *(PCMMSObject*)arg1;
	CMMSObject* obj2 = *(PCMMSObject*)arg2;

	return strcmp((LPCSTR)obj1->sID, (LPCSTR)obj2->sID);
}

int PropsCompareByOrder( const void *arg1, const void *arg2 )
{
	CMMSObject* obj1 = *(PCMMSObject*)arg1;
	CMMSObject* obj2 = *(PCMMSObject*)arg2;

	if( obj1->Order < obj2->Order ) return -1;
	if( obj1->Order > obj2->Order ) return 1;
	return 0;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSObject methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSObject::CMMSObject(CString& sID, CString& ID)
{
	Type = MMS_Object; 
	this->sID = sID;
	this->ID = atoi((LPCSTR)ID);
}

CMMSObject::~CMMSObject()
{
	DestroyObjects();
}

void CMMSObject::DestroyObjects()
{
	int nProps = GetNProps();
	while( nProps > 0 )
	{
		CMMSObject* obj = GetPropByNum(--nProps);
		if( obj != NULL ) delete obj;
	}
}


CMMSObject* CMMSObject::GetProperty(int ID, bool recursive, CMMSNodeList* pList)
{
	int nProps = GetNProps();
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type != MMS_Property && obj->ID == ID )
		{
			if( pList != NULL ) pList->AddTail(obj);
			return obj;
		}
	}

	if( recursive )
	{
		for( int i = 0; i < nProps; i++ )
		{
			CMMSObject* obj = GetPropByNum(i);
			if( obj->Type != MMS_Property )
			{
				CMMSObject* found = obj->GetProperty(ID, true, pList);
				if( found != NULL )
				{
					if( pList != NULL ) pList->AddHead(obj);
					return found;
				}
			}
		}
	}

	return NULL;
}

CMMSObject* CMMSObject::GetProperty(CString& sID)
{
	int nProps = GetNProps();
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type != MMS_Property && obj->sID == sID )  return obj;
	}

	return NULL;
}

int CMMSObject::FindProperty(CString& sID)
{
	int nProps = GetNProps();
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type != MMS_Property && obj->sID == sID )  return i;
	}

	return -1;
}

CString CMMSObject::HdrAsString()
{
	if( (Type == MMS_Property) || (ID == 0) ) 
	{
		if( sID.GetLength() == 0 ) return ""; //  ,     
		return CString("\"") + sID + "\"";
	}
	
	if( sID.GetLength() == 0 ) return ""; //  ,     

	char buf[50];
	CString IDasStr(itoa(ID, buf, 10));
	CString str;
	add_quoted(str, IDasStr);
	if( sID.GetLength() > 0 && sID != IDasStr)
	{
		add_quoted(str, sID);
	}
	return str;
}

bool CMMSObject::PropsAsString(CString& dest, int Depth)
{
	int NProps = GetNProps();
	
	if( NProps == 0 ) return false;

	CString comma(",");
	CString offset("\r\n");
	str_repeate(offset, "\t", Depth+1);
	bool HasObjects = false;
	for( int i = 0; i < NProps; i++ )
	{
		CMMSObject* prop = GetPropByNum(i);
		if( i > 0 ) dest += comma;
		if( prop->Type != MMS_Property )
		{
			dest += offset;
			HasObjects = true;
		}
		dest += GetPropByNum(i)->AsString(Depth+1);
	}

	return HasObjects;
}

CString CMMSObject::AsString(int Depth)
{
	if( Type == MMS_Property ) return CString("\"") + sID + "\"";

	CString str("{"), brace_offset("\r\n");
	str_repeate(brace_offset, "\t", Depth);

	str += HdrAsString();

	bool HasObjects = false;
	if( GetNProps() > 0 )
	{
		if(  sID.GetLength() > 0 ) str += ",";
		if( PropsAsString(str, Depth) ) HasObjects = true;
	}
	
	if( HasObjects && Depth >= 0 ) str += brace_offset;
	str += "}";

	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSObjectWithProps methods  ///////////////////////
///////////////////////////////////////////////////////////////////////
CMMSObjectWithProps::CMMSObjectWithProps(CMMSNodeList *pList)
{
	SetProperties(pList);
}

CMMSObjectWithProps::CMMSObjectWithProps(CString& sID, CString& ID, CMMSNodeList *pList)
:CMMSObject(sID, ID)
{
	SetProperties(pList);
}

CMMSObjectWithProps::~CMMSObjectWithProps()
{
	DestroyObjects();
}

void CMMSObjectWithProps::SetProperties(CMMSNodeList *pList)
{
	if( pList == NULL )
	{
		nProps = 0;
		return;
	}

	pList->DestroyObjects = false;

	Properties.SetSize(pList->GetCount());
	POSITION pos = pList->GetHeadPosition();
	nProps = 0;
	while( pos != NULL )
	{
		CMMSObject* obj = pList->GetNext(pos);
		obj->Order = nProps;
		Properties.SetAt(nProps, obj);
		nProps++;
	}
}

bool CMMSObjectWithProps::DelProperty(int i) 
{
	if( nProps == 0 ) return false;
	Properties.RemoveAt(i);
	nProps--;
	return true;
}

void CMMSObjectWithProps::SortByName()
{
	qsort(Properties.GetData(), nProps, sizeof(CMMSObject*), PropsCompareByName);
}

void CMMSObjectWithProps::SortByOrder()
{
	qsort(Properties.GetData(), nProps, sizeof(CMMSObject*), PropsCompareByOrder);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSBasicObject methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSBasicObject::CMMSBasicObject(CString& sID, CString& ID, CString& Comment, CString& Synonym, CMMSNodeList *pList)
:CMMSObjectWithProps(sID, ID, pList)
{
	Type = MMS_Object;
	this->Comment = Comment;
	this->Synonym = Synonym;
}

/*
CMMSBasicObject::~CMMSBasicObject()
{
	DestroyObjects();
}
*/

CString CMMSBasicObject::HdrAsString()
{
	CString str;
	str.Format("\"%i\",\"%s\",\"%s\",\"%s\"", ID, sID, Comment, Synonym);
	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMS methods  //////////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMS::CMMS(CMMSNodeList *pList)
:CMMSObjectWithProps(pList)
{
}

CString CMMS::AsString(int Depth)
{
	CString str("{");
	PropsAsString(str, Depth);
	if( Depth >= 0 )
		str += "\r\n}\r\n";
	else
		str += "}\r\n";
	return str;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDescr methods  /////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSDescr::CMMSDescr(CString& sID, CString& FirstAvailableID, CString& MD_ver, CString& UnknownID)
:CMMSMetadataPart(MMS_MainDataContDef)
{
	ID = 0;
	this->sID = sID;
	this->FirstAvailableID = atoi((LPCSTR)FirstAvailableID);
	this->MD_ver = atoi((LPCSTR)MD_ver);
	this->UnknownID = atoi((LPCSTR)UnknownID);
}

CString CMMSDescr::AsString(int Depth)
{
	CString str;
	str.Format("{\"%s\",\"%i\",\"%i\",\"%i\"}", sID, FirstAvailableID, MD_ver, UnknownID);
	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSSubCnt methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSSbCnt::CMMSSbCnt(CString& sID, CString& ID, CString& Comment, CString& Synonym, CMMSNodeList *pList)
:CMMSBasicObject(sID, ID, Comment, Synonym, pList)
{
	Type = MMS_SbCnt;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSSbCnts methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSSbCnts::CMMSSbCnts(CString sID, CMMSNodeList* pDocList)
:CMMSObjectWithProps(sID, CString(""))
{
	Type = MMS_SbCnts;
	SetProperties(pDocList);
	SortByName();
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocuments methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSDocuments::CMMSDocuments(CString sID, CMMSNodeList* pDocList)
:CMMSObjectWithProps(sID, CString(""), pDocList)
{
	Type = MMS_Documents;
	SortByName();
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocStream methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSDocStream::CMMSDocStream(CString& sID, CString& ID, 
							 CString& Comment, CString& Synonym, 
							 CString& Param1, CString& Param2, 
							 CMMSObject* Registers, CMMSObject* Documents)
:CMMSObject(sID, ID)
{
	Type = MMS_DocumentStream;
	this->Comment = Comment;
	this->Synonym = Synonym;
	this->Param1 = Param1;
	this->Param2 = Param2;
	this->Registers = Registers;
	this->Documents = Documents;
}

CMMSDocStream::~CMMSDocStream()
{
	if( Registers != NULL ) delete Registers;
	if( Documents != NULL ) delete Documents;
}

CString CMMSDocStream::AsString(int Depth)
{
	CString str, comma(","), offset;
	str_repeate(offset, "\t", Depth+1);
	str.Format("{\"%i\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\r\n%s%s,\r\n%s%s}", 
		ID, sID, 
		Comment, Synonym, Param1, Param2,
		offset, Registers->AsString(Depth+1),
		offset, Documents->AsString(Depth+1));

	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSBuh methods  ///////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSBuh::CMMSBuh(CString sID, CMMSNodeList* pList)
:CMMSObjectWithProps(sID, CString(""), pList)
{
	Type = MMS_Buh;

	CMMSObject* Prop0 = GetPropByNum(0);
	BuhID = Prop0->ID;
	int i = FindProp(CString("Form"), Prop0); //   -   
	AccListForms = Prop0->GetPropByNum(i);
	ProvListForms = Prop0->GetPropByNum(i+2);
	
	i = FindProp(CString("OperJournal"), Prop0);
	OperListForms = Prop0->GetPropByNum(i)->GetPropByNum(0);
	i = FindProp(CString("Form"), OperListForms);
	OperListForms = OperListForms->GetPropByNum(i);
}

int CMMSBuh::FindProp(CString& sID, CMMSObject* obj)
{
	int nProps = obj->GetNProps();
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* prop = obj->GetPropByNum(i);
		if( prop->sID == sID ) return i;
	}
	return -1;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSNode methods  //////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSNode::~CMMSNode()
{
	if( Properties != NULL )
	{
		delete Properties;
	}
}

void CMMSNode::AddValue(CString& Value)
{
	CMMSNode *NewNode = new CMMSNode(MMS_Property, Value, Value);
	if( Properties == NULL ) Properties = new CMMSNodeList;
	Properties->AddTail((CMMSObject*&)NewNode);
}

void CMMSNode::AddValue(CMMSObject *Node)
{
	if( Properties == NULL ) Properties = new CMMSNodeList;
	Properties->AddTail((CMMSObject*&)Node);
}

int CMMSNode::GetNProps()
{
	if( Properties == NULL ) return 0;
	return Properties->GetCount();
}

CMMSObject* CMMSNode::GetPropByNum(int i)
{
	POSITION pos = Properties->GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* obj = Properties->GetNext(pos);
		if( i <= 0 ) return obj;
		i--;
	}
	return NULL;
}

CString CMMSNode::HdrAsString()
{
	if( sID.GetLength() == 0 ) return ""; //  ,     
	return CString("\"") + sID + "\"";
}

bool CMMSNode::PropsAsString(CString& dest, int Depth)
{
	CString comma(",");
	CString offset("\r\n");
	str_repeate(offset, "\t", Depth+1);
	bool HasObjects = false;

	int i = 0;
	POSITION pos = Properties->GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* prop = Properties->GetNext(pos);
		if( i > 0 ) dest += comma;
		if( prop->Type != MMS_Property )
		{
			dest += offset;
			HasObjects = true;
		}
		dest += prop->AsString(Depth+1);
		i++;
	}

	return HasObjects;
}



///////////////////////////////////////////////////////////////////////
///////////  class CMMSNodeList methods  //////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSNodeList::~CMMSNodeList()
{
	if( DestroyObjects )
	{
		POSITION pos = GetHeadPosition();
		while( pos != NULL )
		{
			CMMSObject* SubNode = GetNext(pos);
			delete SubNode;
		}
	}
}

void CMMSNodeList::AddNode(CString& Value)
{
	CMMSNode *NewNode = new CMMSNode(MMS_Property, Value, Value);
	AddTail((CMMSObject*&)NewNode);
}

void CMMSNodeList::AddNode(CMMSObject *Node)
{
	AddTail((CMMSObject*&)Node);
}

void CMMSNodeList::AddNode(CMMSNode *Node)
{
	AddTail((CMMSObject*&)Node);
}

