
#include <io.h>
#include "MMS.h"
#include "NameTable.h"
#include "Storage.h"
#include "CommonFunctions.h"
#include "Container.h"
#include "Task.h"

extern CTaskParameters TaskParameters;
extern CNameTable NameTable;


void CreateTypedTextContainerContents(CStorage& TypedTextStorage)
{
	CString CC = "{\"Container.Contents\",{\"MetaDataDescription\",\"MD Programm text\",\"MD Programm text\",\"\"}}\n";
	TypedTextStorage.StreamFromString("Container.Contents", CC);
}

void CreateContainerContents(CStorage& Storage)
{
	CContainer CC;
	CC.Create(Storage, true);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSObjectWithProps methods  ///////////////////////
///////////////////////////////////////////////////////////////////////
bool CMMSObjectWithProps::ImportMoxelPages(CStorage& Storage, CString& SrcDir, CString& ContCont)
{
	bool HavePages = false;
	CString FName, Page, CC;

	CNameTableRecord* nt_rec = NameTable.Find(NT_ObjectOrdering);
	FILE* ordFile = fopen(nt_rec->PrepareFileName(SrcDir), "rt");
	if( ordFile == NULL ) return false;
	char buffer[4096], *file_name, *sID, *suffix;

	while( NULL != fgets(buffer, sizeof(buffer), ordFile) )
	{
		//   .ord
		char *ptr = buffer;
		file_name = ptr; 
		ptr = strchr(ptr, '\t'); if( ptr == NULL ) goto Error;
		*ptr++ = '\0';
		sID = ptr;
		ptr = strchr(ptr, '\t');  if( ptr == NULL ) goto Error;
		*ptr++ = '\0';
		suffix = ptr;
		ptr = strchr(ptr, '\n');  if( ptr == NULL ) goto Error;
		if( ptr != NULL ) *ptr = '\0';
		
		// 
		FName.Format("%s\\%s", SrcDir, file_name);
		Page.Format("Page.%s", suffix);

		if( TaskParameters.NoEmptyMxl ) //  
		{
			long size = GetFileSize(FName);
			if( size == 147 || size == 139 ) continue;
		}
		
		Storage.StreamFromFile(Page, FName);

		CC.Format(",{\"Moxcel.Worksheet\",\"%s\",\"%s\",\"\"}", Page, sID);
		ContCont += CC;

		HavePages = true;
	}
	fclose(ordFile);

	return HavePages;

Error:
	Msg(0, "ERR: File '%s' has wrong format\n", nt_rec->PrepareFileName(SrcDir));
	return HavePages; //    ,  -  .
}

void CMMSObjectWithProps::ImportContainerProfile(CStorage& Storage)
{
	if( TaskParameters.NoProfiles ) return;

	CString CP = "{\r\n\
{\"MoxelName\",\"\",\"\"},\r\n\
{\"MoxelPos\",\"0\",\"\"},\r\n\
{\"UUID\",\"D41D8CD98F00B204E9800998ECF8427E\",\"\"},\r\n\
{\"Entry\",\"1\",\"\"},\r\n\
{\"MoxelNextMode\",\"1\",\"\"}}";

	Storage.StreamFromString("Container.Profile", CP);
}


bool CMMSObjectWithProps::ImportWorkBook(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec_wb = NameTable.Find(NT_Workbook);
	CNameTableRecord* nt_rec;
	CString ContCont = "{\"Container.Contents\"";
	bool NotEmpty = false;

	if( TaskParameters.CompoundType == CTaskParameters::external_report )
		ContCont += ",{\"MetaDataHolderContainer\",\"Main MetaData Stream\",\"Main MetaData Stream\",\"\"}";

	if( TaskParameters.CompoundType != CTaskParameters::external_report )
		Storage.Create(nt_rec_wb->StorageName);

	// 
	nt_rec = NameTable.Find(NT_DialogForm);
	if( Storage.StreamFromFile(nt_rec->StorageName, nt_rec->PrepareFileName(SrcDir)) )
	{
		NotEmpty = true;
		ContCont += ",{\"DialogEditor\",\"Dialog Stream\",\"Dialog Form\",\"\"}";
	}
	// 
	nt_rec = NameTable.Find(NT_FormModule);
	if( Storage.StreamFromFile(nt_rec->StorageName, nt_rec->PrepareFileName(SrcDir)) )
	{
		NotEmpty = true;
		ContCont += ",{\"TextDocument\",\"MD Programm text\",\"Module text\",\"\"}";
	}

	if( TaskParameters.CompoundType == CTaskParameters::external_report )
	{
		ImportStreamByType(Storage, NT_ErtUserHelp, SrcDir);
		ContCont += ",{\"MetaDataDescription\",\"Inplace description\",\"\",\"\"}";
	}

	// 
	if( ImportMoxelPages(Storage, SrcDir, ContCont) ) NotEmpty = true;
	
	ContCont += "}\n";
	Storage.StreamFromString("Container.Contents", ContCont);
	
	//   Container.Profile
	ImportContainerProfile(Storage);

	if( TaskParameters.CompoundType != CTaskParameters::external_report )
	{
		Storage.Close();
		if( !NotEmpty ) Storage.Delete(nt_rec_wb->StorageName);
	}

	return NotEmpty;
}

void CMMSObjectWithProps::ImportDescription(CStorage& TypedTextStorage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UserHelp);
	if( !FileExist(nt_rec->PrepareFileName(SrcDir)) ) return;

	nt_rec = NameTable.Find(NT_UserHelpStor);

	TypedTextStorage.Create(nt_rec->PrepareStorageName(ID));
	ImportStreamByType(TypedTextStorage, NT_UserHelp, SrcDir);
	CreateTypedTextContainerContents(TypedTextStorage);

	TypedTextStorage.Close();
}

bool CMMSObjectWithProps::ImportListForms(CMMSObject* Forms, CStorage& Storage, 
										  CString& SrcDir, CString Prefix)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_ListForm);
	CString StorName;
	bool NotEmpty = false;

	for( int i = 0; i < Forms->GetNProps(); i++ )
	{
		CMMSObject* Form = Forms->GetPropByNum(i);
		CString FormDir;
		FormDir.Format("%s\\%s.%s", SrcDir, Form->GetPropByNum(0)->sID, nt_rec->Ext);
		
		StorName.Format("%s%i", Prefix, Form->ID);
		Storage.Create(StorName);
		NotEmpty = ImportWorkBook(Storage, FormDir);
		Storage.Close();
		if( !NotEmpty ) Storage.Delete(StorName);
	}

	return NotEmpty;
}

bool CMMSObjectWithProps::ImportStreamByType(CStorage& Storage, MMSObjectType Type, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString FileName = nt_rec->PrepareFileName(SrcDir);

	if( !FileExist(FileName) ) return false;

	CStorage SubStorage(Storage);
	bool res = SubStorage.StreamFromFile(
		nt_rec->PrepareStorageName(ID), 
		FileName);
	
	return res;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMS methods  //////////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMS::ImportStorage(CStorage& Storage, CString& SrcDir, MMSObjectType Type)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage DestStor(Storage);

	DestStor.StorageFromDir(nt_rec->PrepareStorageName(ID), nt_rec->PrepareDirName(SrcDir));
}

void CMMS::ImportCommonMXL(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec_gd = NameTable.Find(NT_GlobalData);
	CNameTableRecord* nt_rec = NameTable.Find(NT_CommonMXL);
	
	CStorage GDStorage(Storage);
	GDStorage.Create(nt_rec_gd->StorageName);
	CStorage MXLStorage(GDStorage);
	MXLStorage.Create(nt_rec->StorageName);

	if( !ImportWorkBook(MXLStorage, nt_rec->PrepareDirName(SrcDir)) )
	{
		MXLStorage.Close();
		GDStorage.Delete(nt_rec->StorageName);
	}
}

void CMMS::ImportGlobalModule(CStorage& Storage, CString& Dir)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_GlobalModuleStor);
	ImportStreamByType(Storage, NT_GlobalModule, Dir);
	CStorage GMStorage(Storage);
	GMStorage.Open(nt_rec->StorageName);
	CreateTypedTextContainerContents(GMStorage);
}

void CMMS::ImportDescriptions(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UserHelpStor);
	CNameTableRecord* nt_rec_file = NameTable.Find(NT_UserHelp);
	CString DescrDir, FileName, FullFName, Pattern, StorageName;
	CString CC = "Container.Contents";

	DescrDir.Format("%s\\%s", SrcDir, nt_rec->Dir);
	Pattern.Format("%s\\*.%s", DescrDir, nt_rec_file->Ext);
	
	Storage.Open("TypedText");
	struct _finddata_t find_data;
	long hFind;
	hFind = _findfirst(Pattern, &find_data);
	if( hFind != -1 )
	{
		do
		{
			FileName = find_data.name;
			int point_pos = FileName.Find('.');
			if( point_pos <= 0 ) continue;

			StorageName.Format("%s%s", nt_rec->StorageName, FileName.Left(point_pos));
			if( !Storage.StorageExist(StorageName) )
			{
				Storage.Create(StorageName);
				FullFName.Format("%s\\%s", DescrDir, find_data.name);
				Storage.StreamFromFile(nt_rec_file->StorageName, FullFName);
				CreateTypedTextContainerContents(Storage);
				Storage.Close();
			}
		} while( _findnext(hFind, &find_data) == 0 );
		_findclose(hFind);
	}
	Storage.Close();
}

void CMMSObjectWithProps::ImportOrdering(CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_ObjectOrdering);
	CString FName = nt_rec->PrepareFileName(SrcDir);
	FILE *File = fopen(FName, "rt");

	if( File == NULL ) return;
	
	char str[4096];
	int ID, i = 0;
	while( !feof(File) )
	{
		fscanf(File, "%i\t%s\n", &ID, str);
		CMMSObject* obj = GetProperty(ID, false);
		if( obj != NULL ) obj->Order = i++;
	}
	fclose(File);

	SortByOrder();
}

void CMMS::Compile(CStorage& Storage, CString& Dir)
{
	if( TaskParameters.CompoundType == CTaskParameters::external_report	)
	{
		ImportWorkBook(Storage, Dir);
		ImportStorage(Storage, Dir, NT_PictureGallery);
		CreateContainerContents(Storage);
		return;
	}

	
	Storage.Create(CString("TypedText"));
	Storage.Close();

	for( int i = 0; i < GetNProps(); i++ )
	{
		CMMSObject *obj = GetPropByNum(i);
		Msg(2, "%s:\n", obj->sID);
		obj->Compile(Storage, Dir);
	}

	// 
	ImportGlobalModule(Storage, Dir);
	// 
	ImportCommonMXL(Storage, Dir);
	//
	ImportStorage(Storage, Dir, NT_PictureGallery);
	//    
	ImportDescriptions(Storage, Dir);
	//
	ImportStorage(Storage, Dir, NT_UserInterface);
	//
	ImportStorage(Storage, Dir, NT_UserRights);

	// 
	ImportStreamByType(Storage, NT_GUIDData, Dir);
	ImportStreamByType(Storage, NT_TagStream, Dir);

	//Container.Contents   UserDef
	Storage.Open("UserDef");
	CString CC = "{\"Container.Contents\",{\"WorkPlaceType\",\"Page.1\",\"\",\"\"},{\"RigthType\",\"Page.2\",\"\",\"\"}}";
	Storage.StreamFromString("Container.Contents", CC);
	Storage.Close();

	//   storage   Container.Contents    .
	CreateContainerContents(Storage);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSSbCnt methods  /////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSSbCnt::Compile(CStorage& Storage, CString& Dir, CStorage& TypedTextStorage, CStorage& SubFolderStorage, CStorage& SubListStorage)
{
	CNameTableRecord* nt_rec_sc = NameTable.Find(Type);
	CNameTableRecord* nt_rec;
	CString ScDir;
	ScDir.Format("%s\\%s", Dir, sID);

	Storage.Create(nt_rec_sc->PrepareStorageName(ID));

	// 
	bool NotEmpty = ImportWorkBook(Storage, ScDir);
	Storage.Close();
	if( !NotEmpty ) Storage.Delete(nt_rec_sc->PrepareStorageName(ID));

	//
	ImportDescription(TypedTextStorage, ScDir);
	// 
	nt_rec = NameTable.Find(NT_SubcontoFolder);
	CString SubFldDir = nt_rec->PrepareDirName(ScDir);
	if( DirectoryExist(SubFldDir) )
	{
		SubFolderStorage.Create(nt_rec->PrepareStorageName(ID));
		ImportWorkBook(SubFolderStorage, SubFldDir);
		SubFolderStorage.Close();
	}

	// 
	nt_rec = NameTable.Find(NT_SubcontoListForm);
	CMMSObject* ListForms = GetProperty(CString("Form"));
	ImportListForms(ListForms, SubListStorage, ScDir, nt_rec->StorageName);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSSbCnts methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSSbCnts::Compile(CStorage& Storage, CString& Dir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage);
	CStorage SubFolderStorage(Storage);
	CStorage SubListStorage(Storage);
	CString ScsDir;
	
	ScsDir.Format("%s\\%s", Dir, nt_rec->Dir);

	TypedTextStorage.Open("TypedText");
	SubFolderStorage.Create(CString("SubFolder"));
	SubListStorage.Create(CString("SubList"));

	Storage.Create(nt_rec->StorageName);
	for( int i = 0; i < GetNProps(); i++ )
	{
		CMMSSbCnt* Sc = (CMMSSbCnt*)GetPropByNum(i);
		Sc->Compile(Storage, ScsDir, TypedTextStorage, SubFolderStorage, SubListStorage);
	}

	Storage.Close();

	ImportOrdering(ScsDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocument methods  //////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSDocument::Compile(CStorage& Storage, CString& SrcDir, CStorage& TypedTextStorage)
{
	CNameTableRecord* nt_rec_stor = NameTable.Find(Type);
	CStorage TransactStorage(TypedTextStorage);
	CString DocDir(SrcDir);

	DocDir += "\\";
	DocDir += sID;

	Storage.Create(nt_rec_stor->PrepareStorageName(ID));
	bool NotEmpty = ImportWorkBook(Storage, DocDir);
	Storage.Close();
	if( !NotEmpty ) Storage.Delete(nt_rec_stor->PrepareStorageName(ID));

	// 
	CNameTableRecord* nt_rec_file = NameTable.Find(NT_TransactModule);
	CString FileName = nt_rec_file->PrepareFileName(DocDir);
	if( FileExist(FileName) )
	{
		nt_rec_stor = NameTable.Find(NT_TransModuleStor);
		TransactStorage.Create(nt_rec_stor->PrepareStorageName(ID));
		ImportStreamByType(TransactStorage, NT_TransactModule, DocDir);
		CreateTypedTextContainerContents(TransactStorage);
	}

	//
	ImportDescription(TypedTextStorage, DocDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocuments methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSDocuments::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString DocsDir = nt_rec->PrepareDirName(SrcDir);

	Storage.Create(nt_rec->PrepareStorageName(ID));
	
	for( int i = 0; i < nProps; i++ )
	{
		CMMSDocument* pDoc = (CMMSDocument*)Properties[i];
		pDoc->Compile(Storage, DocsDir, TypedTextStorage);
	}
	
	Storage.Close();

	ImportOrdering(DocsDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSReport methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSReport::Compile(CStorage& Storage, CString& SrcDir, CString& SubStroragePrefix, CStorage& TypedTextStorage)
{
	CString RptDir(SrcDir);
	CString StorName;

	RptDir += "\\";
	RptDir += sID;

	StorName.Format("%s%i", SubStroragePrefix, ID);
	Storage.Create(StorName);
	ImportWorkBook(Storage, RptDir);
	Storage.Close();

	ImportDescription(TypedTextStorage, RptDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSReportList methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSReportList::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(sID);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString ReportsDir = nt_rec->PrepareDirName(SrcDir);

	Storage.Create(nt_rec->StorageName);
	CString ReportStorName = nt_rec->StorageName + "_Number";
	for( int i = 0; i < nProps; i++ )
	{
		CMMSReport* rpt = (CMMSReport*)Properties[i];
		rpt->Compile(Storage, ReportsDir, ReportStorName, TypedTextStorage);
	}
	Storage.Close();

	ImportOrdering(ReportsDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSJournalister methods  //////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSJournalister::Compile(CStorage& Storage, CString& SrcDir, CStorage& TypedTextStorage)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString JournalDir = SrcDir;
	CMMSObject* Forms = GetProperty(CString("Form"));

	JournalDir += "\\";
	JournalDir += sID;

	ImportDescription(TypedTextStorage, JournalDir);
	ImportListForms(Forms, Storage, JournalDir, nt_rec->StorageName);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSJournalisters methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSJournalisters::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString JournalsDir = nt_rec->PrepareDirName(SrcDir);

	Storage.Create(nt_rec->StorageName);
	for( int i = 0; i < nProps; i++ )
	{
		CMMSJournalister* pJournal = (CMMSJournalister*)Properties[i];
		pJournal->Compile(Storage, JournalsDir, TypedTextStorage);
	}

	Storage.Close();

	ImportOrdering(JournalsDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCalcJournal methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCalcJournal::Compile(CStorage& Storage, CString& SrcDir, CStorage& TypedTextStorage)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString JournalDir(SrcDir);
	CMMSObject* Forms = GetProperty(CString("Form"));

	JournalDir += "\\";
	JournalDir += sID;

	ImportDescription(TypedTextStorage, JournalDir);
	ImportListForms(Forms, Storage, JournalDir, nt_rec->StorageName);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSCalcJournals methods  //////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCalcJournals::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString JournalsDir = nt_rec->PrepareDirName(SrcDir);

	Storage.Create(nt_rec->StorageName);
	for( int i = 0; i < nProps; i++ )
	{
		CMMSCalcJournal* pJournal = (CMMSCalcJournal*)Properties[i];
		pJournal->Compile(Storage, JournalsDir, TypedTextStorage);
	}
	Storage.Close();
	ImportOrdering(JournalsDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCaclAlgorithm methods  /////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCaclAlgorithm::Compile(CStorage& TypedTextStorage, CString& SrcDir)
{
	CNameTableRecord* nt_rec_stor = NameTable.Find(Type);
	CNameTableRecord* nt_rec_file = NameTable.Find(NT_CalcAlgorithm);
	CString AlgDir(SrcDir), FileName;

	AlgDir += "\\";
	AlgDir += sID;

	FileName = nt_rec_file->PrepareFileName(AlgDir);
	if( FileExist(FileName) )
	{
		TypedTextStorage.Create(nt_rec_stor->PrepareStorageName(ID));
		TypedTextStorage.StreamFromFile(nt_rec_file->StorageName, FileName);
		
		CreateTypedTextContainerContents(TypedTextStorage);
		TypedTextStorage.Close();
	}

	ImportDescription(TypedTextStorage, AlgDir);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCaclAlgorithms methods  ////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCaclAlgorithms::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString AlgsDir = nt_rec->PrepareDirName(SrcDir);

	Storage.Open(nt_rec->StorageName); //TypedText already created, we only have to open it
	for( int i = 0; i < nProps; i++ )
	{
		CMMSCaclAlgorithm* pAlg = (CMMSCaclAlgorithm*)Properties[i];
		pAlg->Compile(Storage, AlgsDir);
	}
	Storage.Close();
	ImportOrdering(AlgsDir);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSBuh methods  ///////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSBuh::ImportForm(CStorage& Storage, CString& SrcDir, MMSObjectType Type)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage FormStor(Storage);
	FormStor.Create(nt_rec->PrepareStorageName(BuhID));
	ImportWorkBook(FormStor, nt_rec->PrepareDirName(SrcDir));
}

void CMMSBuh::ImportListForm(CStorage& Storage, CString& SrcDir, MMSObjectType Type, CMMSObject* ListForms)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage ListStor(Storage);
	ListStor.Create(nt_rec->StorageName);
	ImportListForms(ListForms, ListStor, nt_rec->PrepareDirName(SrcDir), nt_rec->StorageName+"_Number");
}

void CMMSBuh::Compile(CStorage& Storage, CString& SrcDir)
{
	// 
	ImportForm(Storage, SrcDir, NT_AccForm);
	// 
	ImportForm(Storage, SrcDir, NT_OperForm);

	//  
	ImportListForm(Storage, SrcDir, NT_AccListForm, AccListForms);
	// 
	ImportListForm(Storage, SrcDir, NT_ProvListForm, ProvListForms);
	// 
	ImportListForm(Storage, SrcDir, NT_OperListForm, OperListForms);
}
