
#include <stdio.h>
#include <io.h>
#include "Storage.h"
#include "CommonFunctions.h"
#include "MMS.h"

#define BUFFER_SIZE 256

#import "zlibeng.dll" no_namespace

//extern CTaskParameters TaskParameters;
static CMapStringToPtr ExtractedFiles;


void AddToExtractedList(CString& FullFName)
{
	ExtractedFiles.SetAt((LPCSTR)FullFName, NULL);
}

void CheckPrune(CString& FullFName)
{
	void *ptr;
	if( !ExtractedFiles.Lookup((LPCSTR)FullFName, ptr) )
	{
		remove((LPCSTR)FullFName);
		Msg(1, "DEL: '%s'\n", (LPCSTR)FullFName);
	}
}

LPCSTR GetStgErrorString(HRESULT err)
{
	switch( err )
	{
		case STG_E_ACCESSDENIED:
			return "Access denied";
		case STG_E_FILENOTFOUND:
			return "The element with the specified name does not exist";
		case STG_E_FILEALREADYEXISTS:
			return "File already exist";
		case STG_E_INVALIDPOINTER:
			return "The pointer specified for the element was invalid";
		case STG_E_INSUFFICIENTMEMORY:
			return "Insufficient memory";
		case STG_E_INVALIDNAME:
			return "Invalid name";
		case STG_E_TOOMANYOPENFILES:
			return "Too many open files";
	}
	return "Unknown error";
}


////////////////////////////////////////////////////////////////////
CStorage::CStorage()
{
	ReleaseRootStorage = true;
	pStream = NULL;
}

CStorage::CStorage(CString& CompoundName)
{
	IStorage* pRootStorage;
	HRESULT hr;
	wchar_t* wCompoundName = new wchar_t[CompoundName.GetLength()+1];
	
	swprintf(wCompoundName , L"%S", (LPCSTR)CompoundName);
	hr = StgCreateDocfile(wCompoundName, 
		STGM_DIRECT | STGM_READWRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE, 0, &pRootStorage);
	delete[] wCompoundName;
	
	if( hr != S_OK )
	{
		Msg(0, "ERR: can't create storage '%s' - %s\n", (LPCSTR)CompoundName, GetStgErrorString(hr));
		exit(1);
	}

	Storages.AddTail(pRootStorage);
	pStream = NULL;
	ReleaseRootStorage = true;
}

CStorage::CStorage(IStorage* pStorage)
{
	ReleaseRootStorage = false;
	Storages.AddTail(pStorage);
	pStream = NULL;
}

CStorage::CStorage(CStorage& Storage)
{
	ReleaseRootStorage = false;
	IStorage* stor = Storage.GetStorage();
	if( stor != NULL ) Storages.AddTail(stor);
	pStream = NULL;
}

CStorage::CStorage(CStorage& Storage, CString SubStorage)
{
	ReleaseRootStorage = false;
	IStorage* stor = Storage.GetStorage();
	if( stor != NULL ) Storages.AddTail(stor);
	pStream = NULL;
	Open(SubStorage);
}

CStorage::~CStorage()
{
	CloseStream();

	POSITION pos = Storages.GetTailPosition();
	while( (pos = Storages.GetTailPosition()) != NULL )
	{
		IStorage* Storage = Storages.GetAt(pos);
		if( (Storages.GetCount() > 1) || ReleaseRootStorage ) Storage->Release();
		Storages.RemoveTail();
	}
}


IStorage* CStorage::GetStorage()
{
	POSITION pos = Storages.GetTailPosition();
	if( pos != NULL ) 
		return Storages.GetAt(pos);
	else
		return NULL;
}

CString CStorage::FullPath()
{
	CString str;
	POSITION pos = Storages.GetHeadPosition();
	while( pos != NULL )
	{
		STATSTG  stat;
		IStorage* Storage = Storages.GetAt(pos);
		CString stor;
		Storage->Stat(&stat, STATFLAG_DEFAULT);
		stor.Format("\\%S", stat.pwcsName);
		str += stor;
		Storages.GetNext(pos);
	}
	return str;
}

bool CStorage::Open(CStringArray* StorageName)
{
	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	int i, max_i = StorageName->GetSize() - 1;
	for( i = 0; i <= max_i; i++ )
	{
		Parent = OpenStorage(Parent, StorageName->GetAt(i));
		if( Parent == NULL ) return false;
		Storages.AddTail(Parent);
	}

	return true;
}

bool CStorage::Open(CString& StorageName)
{
	if( Storages.GetCount() == 0 )
	{
		IStorage* pRootStorage = OpenCompound(StorageName);
		if( pRootStorage == NULL ) return false;
		Storages.AddTail(pRootStorage);
		ReleaseRootStorage = true;
		return true;
	}

	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	if( StorageName.Find("\\") == -1 )
	{
		IStorage* stor = OpenStorage(Parent, StorageName);
		if( stor == NULL ) return false;
		Storages.AddTail(stor);
	}
	else
	{
		CStringArray* name_parts = SplitPath(StorageName);
		bool res = Open(name_parts);
		delete name_parts;
		return res;
	}

	return true;
}

bool CStorage::Open(const wchar_t* StorageName)
{
	CString sStorageName;
	sStorageName.Format("%S", StorageName);
	return Open(sStorageName);
}

bool CStorage::Create(CStringArray* name_parts)
{
	for( int i = 0; i < name_parts->GetSize(); i++ )
	{
		if( !Create(name_parts->GetAt(i)) ) return false;
	}
	return true;
}

bool CStorage::Create(CString& StorageName)
{
	if( StorageName.Find("\\") >= 0 )
	{
		CStringArray* name_parts = SplitPath(StorageName);
		bool res = Create(name_parts);
		delete name_parts;
		return res;
	}

	if( Open(StorageName) ) return true;

	IStorage* pStorage;
	HRESULT hr;
	wchar_t* wStorageName = new wchar_t[StorageName.GetLength()+1];
	swprintf(wStorageName, L"%S", (LPCSTR)StorageName);

	hr = GetStorage()->CreateStorage(wStorageName,
		STGM_DIRECT | STGM_READWRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE, 0,0, 
		&pStorage);
	delete[] wStorageName;

	if( hr != S_OK )
	{
		Msg(0, "ERR: Cant create storage '%s' in '%s' - %s\n", (LPCSTR)StorageName, (LPCSTR)FullPath(), GetStgErrorString(hr));
		return false;
	}

	Storages.AddTail(pStorage);
	return true;
}

bool CStorage::CreateStream(CString& StreamName)
{
	CloseStream();

	if( StreamName.Find("\\") >= 0 )
	{
		CStringArray* name_parts = SplitPath(StreamName);
		this->StreamName = name_parts->GetAt(name_parts->GetSize()-1);

		name_parts->SetSize(name_parts->GetSize()-1);
		bool res = Create(name_parts);
		delete name_parts;
		if( !res ) return false;
	}
	else
		this->StreamName = StreamName;

	HRESULT hr;
	wchar_t* wStreamName = new wchar_t[this->StreamName.GetLength()+1];
	swprintf(wStreamName, L"%S", (LPCSTR)(this->StreamName));


	hr = GetStorage()->CreateStream(wStreamName, 
		STGM_DIRECT | STGM_READWRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE,
		0,0, &pStream);
	delete[] wStreamName;
	
	if( S_OK != hr )
	{
		Msg(0, "ERR: Can't create stream '%s' - %s\n", (LPCSTR)StreamName, GetStgErrorString(hr));
		exit(1);
	}

	StreamInfo.UpdateInfo((char*)(LPCSTR)(this->StreamName));
	return true;
}

bool CStorage::Delete(CString& StorageName)
{
	wchar_t* wStorageName = new wchar_t[StorageName.GetLength()+1];
	HRESULT hr;
	
	swprintf(wStorageName, L"%S", (LPCSTR)StorageName);
	hr = GetStorage()->DestroyElement(wStorageName);
	delete[] wStorageName;

	if( S_OK != hr )
	{
		Msg(0, "ERR: Can't delete element '%s' - %s\n", (LPCSTR)StorageName, GetStgErrorString(hr));
		exit(1);
	}
	return true;
}

void CStorage::Close()
{
	CloseStream();

	POSITION pos = Storages.GetTailPosition();
	if( pos != NULL )
	{
		IStorage* Storage = Storages.GetAt(pos);
		Storage->Release();
		Storages.RemoveTail();
	}
}

void CStorage::CloseStream()
{
	if( pStream != NULL )
	{
		pStream->Release();
		pStream = NULL;
	}
}

bool CStorage::OpenStream(CString& StreamName)
{
	CloseStream();

	IStorage* Parent = GetStorage();
	if( Parent == NULL ) return false;

	if( StreamName.Find("\\") == -1 )
	{
		pStream = ::OpenStream(Parent, StreamName);
		this->StreamName = StreamName;
	}
	else
	{
		CStringArray* name_parts = SplitPath(StreamName);
		CString RealName = name_parts->GetAt(name_parts->GetSize()-1);
		name_parts->SetSize(name_parts->GetSize()-1);
		if( !Open(name_parts) ) return false;
		delete name_parts;
		pStream = ::OpenStream(GetStorage(), RealName);
		this->StreamName = RealName;
	}

	if( pStream == NULL ) return false;
	
	StreamInfo.UpdateInfo((char*)(LPCSTR)(this->StreamName));

	return true;
}

bool CStorage::StorageExist(CString& StorageName)
{
	IStorage* pStorage = ::OpenStorage(GetStorage(), StorageName);
	if( pStorage != NULL )
	{
		pStorage->Release();
		return true;
	}
	return false;
}

bool CStorage::StreamExist(CString& StreamName)
{
	IStream* pStream = ::OpenStream(GetStorage(), StreamName);
	if( pStream != NULL )
	{
		pStream->Release();
		return true;
	}
	return false;
}

long CStorage::GetStreamSize(CString& StreamName)
{
	STATSTG stat;
	long size;
	if( !OpenStream(StreamName) ) return -1L;
	size = ::GetStreamSize(pStream).LowPart;
	CloseStream();
	return size;
}

void CStorage::Rewind()
{
	LARGE_INTEGER SeekZero = {0, 0};
	pStream->Seek(SeekZero, 0, NULL);
}

void CStorage::GetCleanStream()
{
	if( StreamInfo.Packed )
	{
		IStream* pUnpackedStream;
		CreateStreamOnHGlobal(NULL, TRUE, &pUnpackedStream);

		IzlibEnginePtr iLibEngine;
		iLibEngine.CreateInstance(L"V75.zlibEngine");
		iLibEngine->pkDecompress(pStream, pUnpackedStream);

		pStream->Release();
		pStream = pUnpackedStream;
		StreamInfo.ReadSize(pStream); //   
	}
	else if( StreamInfo.HaveSize )
	{
		StreamInfo.ReadSize(pStream); //   ,      
		
		LARGE_INTEGER Offset = {0, 0};
		Offset.LowPart = StreamInfo.ContentOffset;
		pStream->Seek(Offset, STREAM_SEEK_SET, NULL);
		
		IStream* CleanStream;
		CreateStreamOnHGlobal(NULL, TRUE, &CleanStream);
		
		ULARGE_INTEGER BytesToCopy = {0, 0};
		BytesToCopy.LowPart = StreamInfo.Size;
		pStream->CopyTo(CleanStream, BytesToCopy, NULL, NULL);

		Offset.LowPart = 0;
		CleanStream->Seek(Offset, 0, NULL);

		pStream->Release();
		pStream = CleanStream;
	}
	else
	{
		StreamInfo.ReadSize(pStream);
	}

}

bool CStorage::CopyToFile(FILE* File)
{
	void* buff[BUFFER_SIZE];
	unsigned long BytesRead = 0;
	DWORD BytesWritten = 0;

	Rewind();

	do
	{
		pStream->Read(buff, BUFFER_SIZE, &BytesRead);
		BytesWritten = fwrite(buff, 1, BytesRead, File);
		if( BytesWritten < BytesRead ) return false;
	} while( BUFFER_SIZE == BytesRead );
	return true;
}

bool CStorage::CopyToFile(CString& FileName, bool ForceCopy)
{
	if( pStream == NULL ) return false;

	FILE *File;
	char* MsgPrefix = "";

	AddToExtractedList(FileName);

	if( !ForceCopy && FileExist((LPCSTR)FileName) )
	{
		if( !operator==(FileName) ) return true;
		MsgPrefix = "UPD";
	}
	else
	{
		MsgPrefix = "NEW";
	}

	File = fopen((LPCSTR)FileName, "wb");
	if( !CopyToFile(File) )
	{
		Msg(0, "Error writing to file '%s' - %s\n", (LPCSTR)FileName, strerror(errno));
		exit(1);
	}

	fclose(File);

	Msg(1, "%s: %s\n", MsgPrefix, (LPCSTR)FileName);
	return true;
}

bool CStorage::CopyToFile(CString& StreamName, CString& FileName, bool ForceCopy)
{
	if( !OpenStream(StreamName) ) return false;
	GetCleanStream();
	return CopyToFile(FileName, ForceCopy);
}

bool CStorage::Extract(CString& DestDir, bool WithContainerContents)
{
	IEnumSTATSTG *ppenum;
	STATSTG stat;
	unsigned long uCount;
	CString StreamName, FileName, cc("Container.Contents");

	if( !CreateDirectory((LPCSTR)DestDir) ) return false;

	GetStorage()->EnumElements(0, NULL, 0, &ppenum);
	while ( S_OK == ppenum->Next(1, &stat, &uCount) )
	{
		StreamName.Format("%S", stat.pwcsName);
		FileName.Format("%s\\%s", DestDir, StreamName);

		if( (stat.type & STGTY_STORAGE) != 0 )
		{
			if( Open(StreamName) )
			{
				Extract(FileName, WithContainerContents);
				Close();
			}
		}
		else
		{
			if( !WithContainerContents && StreamName == cc ) continue;
			CopyToFile(StreamName, FileName);
		}
	}
	ppenum->Release();

	return true;
}

bool CStorage::FromFile(FILE* File, LPCSTR FileName)
{
	IStream* pStreamText;
	char buff[BUFFER_SIZE];
	size_t BytesRead;

	StreamInfo.ReadFileSize(File); //,   0xFF   
	
	rewind(File);
	CreateStreamOnHGlobal(NULL, TRUE, &pStreamText);
	WriteSizeToStream(pStreamText);

	do
	{
		BytesRead = fread(buff, 1, sizeof(buff), File);
		pStreamText->Write(buff, BytesRead, NULL);
	} while( BytesRead == BUFFER_SIZE );

	if( StreamInfo.Packed )
	{
		IStream* pStreamPack;
		LARGE_INTEGER SeekZero = {0, 0};
		
		pStreamText->Seek(SeekZero, 0, NULL);
		CreateStreamOnHGlobal(NULL, TRUE, &pStreamPack);

		IzlibEnginePtr iLibEngine;
		iLibEngine.CreateInstance(L"V75.zlibEngine");
		iLibEngine->pkCompress(pStreamText, pStreamPack);
		pStreamText->Release();
		pStreamText = pStreamPack;
	}

	ULARGE_INTEGER StreamSize = ::GetStreamSize(pStreamText);
	LARGE_INTEGER SeekZero = {0, 0};
	pStreamText->Seek(SeekZero, 0, NULL);
	pStream->Seek(SeekZero, 0, NULL);
	pStreamText->CopyTo(pStream, StreamSize, NULL, NULL);
	pStreamText->Release();

	if( FileName != NULL ) Msg(2, "CPY: '%s'\n", FileName);

	return true;
}

bool CStorage::FromFile(CString& FileName)
{
	FILE* File;
	File = fopen((LPCSTR)FileName, "rb");
	if( File == NULL ) return false;
	FromFile(File, FileName);
	fclose(File);
	return true;
}

bool CStorage::StreamFromFile(CString& StreamName, CString& FileName)
{
	if( !FileExist(FileName) ) return false;
	if( !CreateStream(StreamName) ) return false;
	return FromFile(FileName);
}

void CStorage::AddZeroByte()
{
	BYTE byte = 0;
	if( pStream == NULL ) return;
	pStream->Write(&byte, 1, NULL);
}

bool CStorage::StreamFromString(CString StreamName, CString& String)
{
	if( !CreateStream(StreamName) ) return false;
	pStream->Write((LPCSTR)String, String.GetLength(), NULL);
	return true;
}

bool CStorage::StorageFromDir(CString& StorageName, CString& SrcDir)
{
	CString FileName;
	struct _finddata_t find_data;
	long hFind;

	if( !Create(StorageName) )
	{
		Msg(0, "ERR: Can't create storage %s\n", StorageName);
		return false;
	}
	//Msg(2, "STOR: %s\n", this->FullPath());

	hFind = _findfirst(SrcDir+"\\*", &find_data);
	if( hFind != -1 )
	{
		do
		{
			if (0 == strcmp(find_data.name, ".")) continue;
			if (0 == strcmp(find_data.name, "..")) continue;
			if (0 == stricmp(find_data.name, "CVS")) continue;
			
			FileName.Format("%s\\%s", SrcDir, find_data.name);

			if( (find_data.attrib & _A_SUBDIR) != 0 )
			{
				StorageFromDir(CString(find_data.name), FileName);
				Close();
			}
			else
				StreamFromFile(CString(find_data.name), FileName);
		} while( _findnext(hFind, &find_data) == 0 );
		_findclose(hFind);
	}

	return true;
}

void CStorage::WriteSizeToStream(IStream* pStream)
{
	BYTE byte_ff = 0xFF;

	if( !StreamInfo.HaveSize ) return;

	//  0xFF
	for( unsigned int i = 0; i < StreamInfo.SizeOffset; i++ ) 
		pStream->Write(&byte_ff, 1, NULL);

	switch( StreamInfo.SizeBytes )
	{
		BYTE bSize;
		WORD wSize;
		DWORD dwSize;
		case 1:
			bSize = (BYTE)StreamInfo.Size;
			pStream->Write(&bSize, StreamInfo.SizeBytes, NULL);
			break;
		case 2:
			wSize = (WORD)StreamInfo.Size;
			pStream->Write(&wSize, StreamInfo.SizeBytes, NULL);
			break;
		case 4:
			dwSize = StreamInfo.Size;
			pStream->Write(&dwSize, StreamInfo.SizeBytes, NULL);
			break;
		default:
			Msg(0, "ERR: Don't know how to write size of stream (%i bytes)\n", (int)StreamInfo.SizeBytes);
			exit(1);
			break;
	}
}



bool CStorage::operator== (CString& FileName)
{
	if( !FileExist((LPCSTR)FileName) ) return true;

	FILE* File;
	long FileSize;
	void* buff_Stream[BUFFER_SIZE];
	void* buff_File[BUFFER_SIZE];
	DWORD BytesReadF = 0, BytesReadS = 0;

	File = fopen((LPCSTR)FileName, "rb");
	if (File == NULL ) 
	{ 
		Msg(0, "ERR: Can\t open file '%s' - %s\n", (LPCSTR)FileName, strerror(errno));
		exit(1);
    }

	fseek(File, 0, SEEK_END);
	FileSize = ftell(File);
	
	Rewind();
	rewind(File);

	//  ?
	Msg(2, "File size is: %d - %s\n", FileSize, FileName);
	Msg(2, "Strm size is: %d\n", StreamInfo.Size);
	if ( FileSize != (StreamInfo.Size) )	goto FilesDiffer;

	//  -    
	do
	{
		BytesReadF = fread(buff_File, 1, BUFFER_SIZE, File);
		pStream->Read(buff_Stream, BUFFER_SIZE, &BytesReadS);
		
		if( BytesReadF != BytesReadS ) goto FilesDiffer;
		if( memcmp(buff_File, buff_Stream, BytesReadF) != 0 ) goto FilesDiffer;

	} while( BUFFER_SIZE == BytesReadF );

	fclose(File);
	return false;

FilesDiffer:
	fclose(File);
	return true;
}

extern int MMS_debug;
CMMSObject* CStorage::ParseStream(CString& StreamName)
{
	FILE* File;
	CMMSObject* MMSObject;

	OpenStream(StreamName);
	GetCleanStream();

	File = tmpfile();
	if( File == NULL )
	{
		Msg(0, "ERR: Can't create temporary file\n");
		exit(1);
	}

	if( !CopyToFile(File) )
	{
		Msg(0, "ERR: Error extracting '%s' into temporary file\n", (LPCSTR)StreamName);
		exit(1);
	}
	rewind(File);

	MMS_InitParser();
	//MMS_debug = 1;
	int RetCode = MMS_parse(File, &MMSObject);
	fclose(File);
	if( RetCode != 0 ) 
	{
		Msg(0, "ERR: Error parsing %\n", (LPCSTR)StreamName);
		Msg(0, "Main Metadata Stream is corrupted, or password protected.");
		exit(1);
	}

	return MMSObject;
}

////////////////////////////////////////////////////////////////////////////////////////
ULARGE_INTEGER GetStreamSize(IStream* pStream)
{
	STATSTG stat;
	pStream->Stat(&stat, STATFLAG_NONAME);
	return stat.cbSize;
}

IStorage* OpenCompound(CString& CompoundFileName)
{
	IStorage* pRootStorage;
	wchar_t* wCompoundFileName = new wchar_t[CompoundFileName.GetLength()+1];

	swprintf(wCompoundFileName, L"%S", (LPCSTR)CompoundFileName);
	HRESULT hr = StgOpenStorage(wCompoundFileName , NULL, 
		STGM_SHARE_EXCLUSIVE | STGM_DIRECT | STGM_READWRITE
		, NULL, 0, 
		&pRootStorage);

	delete[] wCompoundFileName;

	if( S_OK != hr )
	{
		Msg(0, "ERR: Can not open storage '%s' - %s\n", 
			(LPCSTR)CompoundFileName,  GetStgErrorString(hr));
		exit(1);
	}

	return pRootStorage;
}

IStorage* OpenStorage(IStorage* pParentStorage, wchar_t* wStorName)
{
	IStorage* pStorage;
	pParentStorage->OpenStorage(wStorName, NULL, STGM_SHARE_EXCLUSIVE | STGM_READWRITE, NULL, 0, &pStorage);
	return pStorage;
}

IStorage* OpenStorage(IStorage* pParentStorage, CString& StorName)
{
	IStorage* pStorage;

	wchar_t* wStorName = new wchar_t[StorName.GetLength()+1];
	swprintf(wStorName, L"%S", (LPCSTR)StorName);
	pStorage = OpenStorage(pParentStorage, wStorName);
	delete[] wStorName;

	return pStorage;
}

IStream* OpenStream(IStorage* pParentStorage, wchar_t* wStreamName)
{
	IStream* pStream;
	pParentStorage->OpenStream(wStreamName, NULL, STGM_SHARE_EXCLUSIVE, 0, &pStream);
	return pStream;
}

IStream* OpenStream(IStorage* pParentStorage, CString& StreamName)
{
	IStream* pStream;
	
	wchar_t* wStreamName = new wchar_t[StreamName.GetLength()+1];
	swprintf(wStreamName, L"%S", (LPCSTR)StreamName);
	pStream = OpenStream(pParentStorage, wStreamName);
	delete[] wStreamName;

	return pStream;
}


bool StringDifferFromFile(CString& String, CString& FileName)
{
	if( !FileExist((LPCSTR)FileName) ) return true;

	FILE* File = fopen((LPCSTR)FileName, "rb");
	if (File == NULL ) 
	{ 
		Msg(0, "ERR: Can\t open file '%s' - %s\n", (LPCSTR)FileName, strerror(errno));
		exit(1);
    }

	long FileSize;
	long BytesRead;
	void* buff = NULL;
	
	fseek(File, 0, SEEK_END);
	FileSize = ftell(File);
	rewind(File);

	if ( String.GetLength() != FileSize ) goto FilesDiffer;

	buff = (void*) new char[FileSize];
	BytesRead = fread(buff, 1, FileSize, File);
	if( memcmp(buff, (LPCSTR)String, FileSize) != 0 ) goto FilesDiffer;

	delete[] buff;
	fclose(File);
	return false;

FilesDiffer:
	if( buff != NULL ) delete[] buff;
	fclose(File);
	return true;
}

bool CopyStringToFile(CString& String, CString& FileName)
{
	char* MsgPrefix;

	AddToExtractedList(FileName);

	//String.Replace("\n", "\r\n");

	if( FileExist(FileName) )
		if( !StringDifferFromFile(String, FileName) )
			return true;
		else
			MsgPrefix = "UPD";
	else
		MsgPrefix = "NEW";

	FILE *File = fopen((LPCSTR)FileName, "wb"); //    ,    .
	fwrite((LPCSTR)String, 1, String.GetLength(), File);
	fclose(File);
	Msg(1, "%s: %s\n", MsgPrefix, (LPCSTR)FileName);

	return true;
}


