// TaskCompile.cpp: implementation of the CTaskCompile class.
//////////////////////////////////////////////////////////////////////

#include <io.h>
#include "Task.h"
#include "Container.h"
#include "CommonFunctions.h"
#include "Storage.h"
#include "MMS.h"
#include "NameTable.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define BUFFER_SIZE 256

extern CTaskParameters TaskParameters;
extern CNameTable NameTable;

//////////////////////////////////////////////////////////////////////
void AddFile(FILE* DestFile, CString& FName, bool AddComma = true)
{
	char buff[BUFFER_SIZE];
	long BytesRead;
	FILE *SrcFile = fopen((LPCSTR)FName, "rb");

	if( SrcFile == NULL )
	{
		Msg(0, "ERR: Can't read file '%s' - %s\n", (LPCSTR)FName, strerror(errno));
		exit(1);
	}

	if( ftell(DestFile) > 0 && AddComma )
	{
		fputs(",\r\n", DestFile);
	}

	do
	{
		BytesRead = fread(buff, 1, sizeof(buff), SrcFile);
		fwrite(buff, 1, BytesRead, DestFile);
	} while( BytesRead == sizeof(buff) );

	fclose(SrcFile);
}

void AddDir(FILE* DestFile, CString& DirName, LPCSTR sID)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_MetadataPart);
	CString FName = nt_rec->PrepareFileName(DirName);

	fputs(",\r\n{\"", DestFile);
	fputs(sID, DestFile);
	fputs("\"\r\n", DestFile);
	if( FileExist(FName) ) AddFile(DestFile, FName);

	struct _finddata_t find_data;
	long hFind;

	hFind = _findfirst(DirName+"\\*", &find_data);
	if( hFind != -1 )
	{
		do
		{
			if (0 == strcmp(find_data.name, ".")) continue;
			if (0 == strcmp(find_data.name, "..")) continue;
			if (0 == stricmp(find_data.name, "CVS")) continue;
			if( (find_data.attrib & _A_SUBDIR) == 0 ) continue;

			FName.Format("%s\\%s\\%s.%s", DirName, find_data.name, nt_rec->FileName, nt_rec->Ext);
			if( !FileExist(FName) ) continue;

			AddFile(DestFile, FName);
		} while( _findnext(hFind, &find_data) == 0 );
		_findclose(hFind);
	}

	fputs("\r\n}", DestFile);
}


FILE* AssembleMMS()
{
	FILE *mmsFile;

	mmsFile = tmpfile();
	if( mmsFile == NULL )
	{
		Msg(0, "ERR: Can't create temporary file\n");
		exit(1);
	}

	if( TaskParameters.CompoundType == CTaskParameters::external_report )
	{
		CNameTableRecord* nt_rec = NameTable.Find(NT_MainMetadataStream);
		AddFile(mmsFile, nt_rec->PrepareFileName(TaskParameters.DirectoryName), false);
	}
	else
	{
		CString Dir = TaskParameters.DirectoryName;
		
		fputs("{\r\n", mmsFile);
		
		bool AddComma = false;
		for( int i = 0; i < NameTable.GetNRecords(); i++ )
		{
			CNameTableRecord *nt_rec = NameTable[i];

			if( nt_rec->sID.GetLength() == 0 ) continue; //    

			if( nt_rec->FileName.GetLength() != 0 )
				AddFile(mmsFile, nt_rec->PrepareFileName(Dir), AddComma);
			else
				AddDir(mmsFile, Dir+"\\"+nt_rec->Dir, nt_rec->sID);

			AddComma = true;
		}

		fputs("\r\n}", mmsFile);
	}


	return mmsFile;
}

char* MinimalMDS()
{
	return "{\r\n\
{\"MainDataContDef\",\"15\",\"10009\",\"7120\"},\r\n\
{\"TaskItem\",\r\n\
{\"1\",\"\",\"\",\"\",\"\",\"1\",\"0\",\"0\",\"1\",\"0\"}},\r\n\
{\"GenJrnlFldDef\"},\r\n\
{\"DocSelRefObj\"},\r\n\
{\"DocNumDef\"},\r\n\
{\"Consts\"},\r\n\
{\"SbCnts\"},\r\n\
{\"Registers\"},\r\n\
{\"Documents\"},\r\n\
{\"Journalisters\"},\r\n\
{\"EnumList\"},\r\n\
{\"ReportList\"},\r\n\
{\"CJ\"},\r\n\
{\"Calendars\",\r\n\
{\"HolidaysDef\",\r\n\
{\"11\",\"\",\"\",\"\"}}},\r\n\
{\"Algorithms\"},\r\n\
{\"RecalcRules\"},\r\n\
{\"CalcVars\"},\r\n\
{\"Groups\"},\r\n\
{\"Document Streams\"},\r\n\
{\"Buh\",\r\n\
{\"2\",\"\",\"\",\"\",\"0\",\"8\",\"8\",\"25\",\"0\",\"0\",\"25\",\"14\",\"2\",\"14\",\"2\",\"14\",\"2\",\"14\",\"3\",\"0\",\"0\",\"10\",\"10\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\",\"1\",\"1\",\"1\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\",\"1\",\"0\",\"1\",\"0\",\"0\",\"0\",\"0\",\"1\",\r\n\
{},\"1\",\"1\",\r\n\
{\"AccParams\"},\r\n\
{\"Plans\"},\r\n\
{\"ProvParams\"},\r\n\
{\"OperParams\"},\r\n\
{\"Form\"},\r\n\
{\"Form\"},\r\n\
{\"Form\"},\r\n\
{\"SbKind\"},\r\n\
{\"TypOpersDef\",\r\n\
{\"3\",\"\",\"\",\"\"}},\r\n\
{\"CorrProvsDef\",\r\n\
{\"4\",\"\",\"\",\"\"}},\r\n\
{\"ProvHardFldDef\",\r\n\
{\"5\",\"\",\"\",\"\",\"P\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\"},\r\n\
{\"6\",\"\",\"\",\"\",\"U\",\"0\",\"0\",\"0\",\"0\",\"0\",\"0\"}},\r\n\
{\"OperJournal\",\r\n\
{\"7\",\"\",\"\",\"\",\"-1\",\"0\",\"9\",\"9\",\"1\",\r\n\
{\"0\",\"\",\"\",\"\",\r\n\
{\"Refers\"}},\r\n\
{\"JournalFld\"},\r\n\
{\"Form\",\r\n\
{\"9\",\"\",\"\",\"\"}}}}}}},\r\n\
";
}


void ImportMMS(CStorage& Storage, FILE* mmsFile, CMMSObject* MMSObject)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_MainMetadataStream);
	FILE *mmsFile_;

	mmsFile_ = tmpfile();
	fseek(mmsFile_, 0, SEEK_SET);

	//  
	CMMSDescr* Descr = (CMMSDescr*)MMSObject->GetPropByNum(0);
	Descr->MD_ver = 10009;
	// CRC
	int crc_num = MMSObject->FindProperty(CString("CRC"));
	if( crc_num >= 0 ) MMSObject->DelProperty(crc_num);

	if( (TaskParameters.CompoundType == CTaskParameters::external_report) 
		&& TaskParameters.TruncateMMS )
	{
		fputs(MinimalMDS(), mmsFile_);
	}
	else
	{
		fputs(MMSObject->AsString(-100), mmsFile_);
	}

	if( TaskParameters.CompoundType == CTaskParameters::external_report )
	{
		Storage.CreateStream(nt_rec->StorageName);
		Storage.FromFile(mmsFile_, nt_rec->StorageName);
		Storage.AddZeroByte(); //        '\0'
	}
	else
	{
		Storage.Create(CString("Metadata"));
		Storage.CreateStream(nt_rec->StorageName);
		Storage.FromFile(mmsFile_, nt_rec->StorageName);
		Storage.Close();
	}

	fclose(mmsFile_);
}


void CTaskCompile::Action()
{
	CTime Time = CTime::GetCurrentTime();

	if( !DirectoryExist(TaskParameters.DirectoryName) )
	{
		Msg(0, "ERR: Source directory '%s' not exist\n", TaskParameters.DirectoryName);
		return;
	}

	FILE *mmsFile = AssembleMMS();
	CMMSObject* MMSObject;
	rewind(mmsFile);
	MMS_InitParser();
	int RetCode = MMS_parse(mmsFile, &MMSObject);

	CStorage Storage(TaskParameters.CompoundFileName);
	if( Storage.GetStorage() != NULL )
	{
		CoInitialize(NULL);
		ImportMMS(Storage, mmsFile, MMSObject);
		MMSObject->Compile(Storage, TaskParameters.DirectoryName);

		CoUninitialize();
	}
	fclose(mmsFile);

	delete MMSObject;

	CTimeSpan CompilationTime = CTime::GetCurrentTime() - Time;
	CString FileName(TaskParameters.CompoundFileName);
	CString DirName(TaskParameters.DirectoryName);
	FileName.AnsiToOem();
	DirName.AnsiToOem();

	Msg(0, "Dir '%s' compiled into '%s' in %s\n", 
		TaskParameters.DirectoryName, TaskParameters.CompoundFileName, 
		CompilationTime.Format("%H:%M:%S"));
}
