
#include <io.h>
#include "MMS.h"
#include "NameTable.h"
#include "Storage.h"
#include "CommonFunctions.h"
#include "Container.h"
#include "Ordering.h"
#include "Task.h"

void CreateTypedTextContainerContents(CStorage& TypedTextStorage)
{
	CString CC = "{\"Container.Contents\",{\"MetaDataDescription\",\"MD Programm text\",\"MD Programm text\",\"\"}}\n";
	TypedTextStorage.StreamFromString("Container.Contents", CC);
}

void CreateContainerContents(CStorage& Storage)
{
	CContainer CC;
	CC.Create(Storage, true);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSObject methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSObject::ImportDescriptions(CStorage& TypedTextStor, CString& FNamePrefix, bool StdName)
{
	if( Type == MMS_Property ) return; //    

	CNameTableRecord* nt_rec = NameTable.Find(NT_UserHelpStor);
	CNameTableRecord* nt_rec_file = NameTable.Find(NT_UserHelp);
	CString StorName = nt_rec->PrepareStorageName(ID);
	CString NewPrefix = FNamePrefix;
	bool to_import = true;
	
	if( ID_Type == IDT_NoID )
		to_import = false;
	else if( ID_Type == IDT_OnlyTextual )
	{
		to_import = false;
		NewPrefix.Format("%s%s.", FNamePrefix, sID);
	}
	else if( ID_Type == IDT_OnlyNumeric )
	{
		to_import = false;
		NewPrefix.Format("%s%i.", FNamePrefix, ID);
	}
	else if( !IsValidFileName(sID) )
		NewPrefix.Format("%s%i.", FNamePrefix, ID);
	else
		NewPrefix.Format("%s%s.", FNamePrefix, sID);

	if( to_import && !TypedTextStor.StorageExist(StorName) )
	{
		CString FName;

		if( StdName )
			FName.Format("%s%s.%s", FNamePrefix, nt_rec_file->FileName, nt_rec_file->Ext);
		else
			FName.Format("%s%s", NewPrefix, nt_rec_file->Ext);

		if( FileExist(FName) )
		{
			TypedTextStor.Create(StorName);
			TypedTextStor.StreamFromFile(nt_rec_file->PrepareStorageName(ID), FName);
			CreateTypedTextContainerContents(TypedTextStor);
			TypedTextStor.Close();
		}
	}

	for( int i = 0; i < GetNProps(); i++ )
	{
		GetPropByNum(i)->ImportDescriptions(TypedTextStor, NewPrefix, false);
	}
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSObjectWithProps methods  ///////////////////////
///////////////////////////////////////////////////////////////////////
bool CMMSObjectWithProps::ImportMoxelPages(CStorage& Storage, CString& SrcDir, CString& ContCont)
{
	bool HavePages = false;
	CString FName, Page, CC;

	CObjectOrdering Order(4);

	if( !Order.OpenFile(SrcDir) ) return false;

	while( Order.ReadString() )
	{
		// 
		FName.Format("%s\\%s", SrcDir, Order.GetValue(2));
		Page.Format("Page.%s", Order.GetValue(0));

		if( TaskParameters.NoEmptyMxl ) //  
		{
			long size = GetFileSize(FName);
			if( size == 147 || size == 139 ) continue;
		}
		
		Storage.StreamFromFile(Page, FName);

		CC.Format(",{\"Moxcel.Worksheet\",\"%s\",\"%s\",\"\"}", Page, Order.GetValue(3));
		ContCont += CC;

		HavePages = true;
	}

	return HavePages;
}

void CMMSObjectWithProps::ImportContainerProfile(CStorage& Storage)
{
	if( TaskParameters.NoProfiles ) return;

	CString CP = "{\r\n\
{\"MoxelName\",\"\",\"\"},\r\n\
{\"MoxelPos\",\"0\",\"\"},\r\n\
{\"UUID\",\"D41D8CD98F00B204E9800998ECF8427E\",\"\"},\r\n\
{\"Entry\",\"1\",\"\"},\r\n\
{\"MoxelNextMode\",\"1\",\"\"}}";

	Storage.StreamFromString("Container.Profile", CP);
}


bool CMMSObjectWithProps::ImportStreamByType(CStorage& Storage, MMSObjectType Type, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString FileName = nt_rec->PrepareFileName(SrcDir);

	if( !FileExist(FileName) ) return false;

	CStorage SubStorage(Storage);
	bool res = SubStorage.StreamFromFile(
		nt_rec->PrepareStorageName(ID), 
		FileName);
	
	return res;
}

bool CMMSObjectWithProps::ImportWorkBook(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec_wb = NameTable.Find(NT_Workbook);
	CNameTableRecord* nt_rec;
	CString ContCont = "{\"Container.Contents\"";
	bool NotEmpty = false;

	//     
	if( TaskParameters.CompoundType == CTaskParameters::external_report )
		ContCont += ",{\"MetaDataHolderContainer\",\"Main MetaData Stream\",\"Main MetaData Stream\",\"\"}";

	if( TaskParameters.CompoundType != CTaskParameters::external_report )
		Storage.Create(nt_rec_wb->StorageName);

	// 
	nt_rec = NameTable.Find(NT_DialogForm);
	if( Storage.StreamFromFile(nt_rec->StorageName, nt_rec->PrepareFileName(SrcDir)) )
	{
		NotEmpty = true;
		ContCont += ",{\"DialogEditor\",\"Dialog Stream\",\"Dialog Form\",\"\"}";
	}
	// 
	nt_rec = NameTable.Find(NT_FormModule);
	if( Storage.StreamFromFile(nt_rec->StorageName, nt_rec->PrepareFileName(SrcDir)) )
	{
		NotEmpty = true;
		ContCont += ",{\"TextDocument\",\"MD Programm text\",\"Module text\",\"\"}";
	}

	//       .   "Inplace description"
	if( TaskParameters.CompoundType == CTaskParameters::external_report )
	{
		ImportStreamByType(Storage, NT_ErtUserHelp, SrcDir);
		ContCont += ",{\"MetaDataDescription\",\"Inplace description\",\"\",\"\"}";
	}

	// 
	if( ImportMoxelPages(Storage, SrcDir, ContCont) ) NotEmpty = true;
	
	ContCont += "}\n";
	Storage.StreamFromString("Container.Contents", ContCont);
	
	//   Container.Profile
	ImportContainerProfile(Storage);

	if( TaskParameters.CompoundType != CTaskParameters::external_report )
	{
		Storage.Close();
		if( !NotEmpty ) Storage.Delete(nt_rec_wb->StorageName);
	}

	return NotEmpty;
}

bool CMMSObjectWithProps::ImportListForms(CMMSObject* Forms, CStorage& Storage, 
										  CString& SrcDir, CString Prefix)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_ListForm);
	CString StorName;
	bool NotEmpty = false;

	for( int i = 0; i < Forms->GetNProps(); i++ )
	{
		CMMSObject* Form = Forms->GetPropByNum(i);
		CString FormDir;
		FormDir.Format("%s\\%s.%s", SrcDir, Form->sID, nt_rec->Ext);
		
		StorName.Format("%s%i", Prefix, Form->ID);
		Storage.Create(StorName);
		NotEmpty = ImportWorkBook(Storage, FormDir);
		Storage.Close();
		if( !NotEmpty ) Storage.Delete(StorName);
	}

	return NotEmpty;
}

void CMMSObjectWithProps::ImportOrdering(CString& SrcDir)
{
	CObjectOrdering Order(3);
	if( !Order.OpenFile(SrcDir) ) return;
	
	int i = 0;
	while( Order.ReadString() )
	{
		int ID = atoi(Order.GetValue(0));
		CMMSObject* obj = GetProperty(ID, false);
		if( obj != NULL ) obj->Order = i++;
	}

	SortByOrder(); // TODO ,     ?
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMS methods  //////////////////////////////////////
///////////////////////////////////////////////////////////////////////
bool CMMS::ImportStorage(CStorage& Storage, CString& SrcDir, MMSObjectType Type)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage DestStor(Storage);

	return DestStor.StorageFromDir(nt_rec->PrepareStorageName(ID), nt_rec->PrepareDirName(SrcDir));
}

void CMMS::ImportCommonMXL(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec_gd = NameTable.Find(NT_GlobalData);
	CNameTableRecord* nt_rec = NameTable.Find(NT_CommonMXL);
	
	CStorage GDStorage(Storage);
	GDStorage.Create(nt_rec_gd->StorageName);
	CStorage MXLStorage(GDStorage);
	MXLStorage.Create(nt_rec->StorageName);

	if( !ImportWorkBook(MXLStorage, nt_rec->PrepareDirName(SrcDir)) )
	{
		MXLStorage.Close();
		GDStorage.Delete(nt_rec->StorageName);
	}
}

void CMMS::ImportGlobalModule(CStorage& Storage, CString& Dir)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_GlobalModuleStor);
	ImportStreamByType(Storage, NT_GlobalModule, Dir);
	CStorage GMStorage(Storage);
	GMStorage.Open(nt_rec->StorageName);
	CreateTypedTextContainerContents(GMStorage);
}


void CMMS::ImportMMS(CStorage& Storage)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_MainMetadataStream);

	//  
	CMMSDescr* Descr = (CMMSDescr*)GetPropByNum(0);
	Descr->MD_ver = 10009;
	// CRC
	int crc_num = FindProperty(CString("CRC"));
	if( crc_num >= 0 ) DelProperty(crc_num);


	if( TaskParameters.CompoundType != CTaskParameters::external_report )
	{
		Storage.Create(CString("Metadata"));
	}

	if( (TaskParameters.CompoundType == CTaskParameters::external_report) 
		&& TaskParameters.TruncateMMS )
	{
		Storage.StreamFromString(nt_rec->StorageName, CString(MinimalMMS()));
	}
	else
	{
		Storage.StreamFromString(nt_rec->StorageName, AsString(-100));
	}

	if( TaskParameters.CompoundType == CTaskParameters::external_report )
	{
		Storage.AddZeroByte(); //        '\0'
	}

	if( TaskParameters.CompoundType != CTaskParameters::external_report )
	{
		Storage.Close();
	}
}


void CMMS::Compile(CStorage& Storage, CString& Dir)
{
	if( TaskParameters.CompoundType == CTaskParameters::external_report	)
	{
		ImportWorkBook(Storage, Dir);
		ImportStorage(Storage, Dir, NT_PictureGallery);
		ImportMMS(Storage);
		CreateContainerContents(Storage);
		return;
	}
	
	//    
	CNameTableRecord* nt_rec_descr = NameTable.Find(NT_UserHelpStor);
	CString DescrDir = nt_rec_descr->PrepareDirName(Dir);
	Storage.Create(CString("TypedText"));
	ImportDescriptions(Storage, DescrDir+"\\", false);
	Storage.Close();

	for( int i = 0; i < GetNProps(); i++ )
	{
		CMMSObject *obj = GetPropByNum(i);
		Msg(2, "%s:\n", obj->sID);
		obj->Compile(Storage, Dir);
	}

	// 
	ImportGlobalModule(Storage, Dir);
	// 
	ImportCommonMXL(Storage, Dir);
	//
	ImportStorage(Storage, Dir, NT_PictureGallery);
	//
	ImportStorage(Storage, Dir, NT_UserInterface);
	//
	ImportStorage(Storage, Dir, NT_UserRights);

	// 
	ImportStreamByType(Storage, NT_GUIDData, Dir);
	ImportStreamByType(Storage, NT_TagStream, Dir);

	//Container.Contents   UserDef
	Storage.Open("UserDef");
	CString CC = "{\"Container.Contents\",{\"WorkPlaceType\",\"Page.1\",\"\",\"\"},{\"RigthType\",\"Page.2\",\"\",\"\"}}";
	Storage.StreamFromString("Container.Contents", CC);
	Storage.Close();

	//  - Main Metadata Stream
	ImportMMS(Storage);

	//   storage   Container.Contents    .
	CreateContainerContents(Storage);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSMetadataPart methods  //////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSMetadataPart::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	if( !nt_rec->Dir.IsEmpty() )
	{
		CString Dir = nt_rec->PrepareDirName(SrcDir);
		CStorage TypedTextStorage(Storage);
		TypedTextStorage.Open("TypedText");
		ImportDescriptions(TypedTextStorage, Dir+"\\", true);
	}
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSSbCnt methods  /////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSSbCnt::Compile(CStorage& Storage, CString& Dir, CStorage& TypedTextStorage, CStorage& SubFolderStorage, CStorage& SubListStorage)
{
	CNameTableRecord* nt_rec_sc = NameTable.Find(Type);
	CNameTableRecord* nt_rec;
	CString ScDir;
	ScDir.Format("%s\\%s", Dir, sID);

	Storage.Create(nt_rec_sc->PrepareStorageName(ID));

	// 
	bool NotEmpty = ImportWorkBook(Storage, ScDir);
	Storage.Close();
	if( !NotEmpty ) Storage.Delete(nt_rec_sc->PrepareStorageName(ID));

	//
	ImportDescriptions(TypedTextStorage, ScDir+"\\", true);
	// 
	nt_rec = NameTable.Find(NT_SubcontoFolder);
	CString SubFldDir = nt_rec->PrepareDirName(ScDir);
	if( DirectoryExist(SubFldDir) )
	{
		SubFolderStorage.Create(nt_rec->PrepareStorageName(ID));
		ImportWorkBook(SubFolderStorage, SubFldDir);
		SubFolderStorage.Close();
	}

	// 
	nt_rec = NameTable.Find(NT_SubcontoListForm);
	CMMSObject* ListForms = GetProperty(CString("Form"));
	ImportListForms(ListForms, SubListStorage, ScDir, nt_rec->StorageName);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSSbCnts methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSSbCnts::Compile(CStorage& Storage, CString& Dir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage);
	CStorage SubFolderStorage(Storage);
	CStorage SubListStorage(Storage);
	CString ScsDir;
	
	ScsDir.Format("%s\\%s", Dir, nt_rec->Dir);

	ImportOrdering(ScsDir);

	TypedTextStorage.Open("TypedText");
	SubFolderStorage.Create(CString("SubFolder"));
	SubListStorage.Create(CString("SubList"));

	Storage.Create(nt_rec->StorageName);
	for( int i = 0; i < GetNProps(); i++ )
	{
		CMMSSbCnt* Sc = (CMMSSbCnt*)GetPropByNum(i);
		Sc->Compile(Storage, ScsDir, TypedTextStorage, SubFolderStorage, SubListStorage);
	}

	Storage.Close();
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocument methods  //////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSDocument::Compile(CStorage& Storage, CString& SrcDir, CStorage& TypedTextStorage)
{
	CNameTableRecord* nt_rec_stor = NameTable.Find(Type);
	CStorage TransactStorage(TypedTextStorage);
	CString DocDir(SrcDir);

	DocDir += "\\";
	DocDir += sID;

	Storage.Create(nt_rec_stor->PrepareStorageName(ID));
	bool NotEmpty = ImportWorkBook(Storage, DocDir);
	Storage.Close();
	if( !NotEmpty ) Storage.Delete(nt_rec_stor->PrepareStorageName(ID));

	// 
	CNameTableRecord* nt_rec_file = NameTable.Find(NT_TransactModule);
	CString FileName = nt_rec_file->PrepareFileName(DocDir);
	if( FileExist(FileName) )
	{
		nt_rec_stor = NameTable.Find(NT_TransModuleStor);
		TransactStorage.Create(nt_rec_stor->PrepareStorageName(ID));
		ImportStreamByType(TransactStorage, NT_TransactModule, DocDir);
		CreateTypedTextContainerContents(TransactStorage);
	}

	//
	ImportDescriptions(TypedTextStorage, DocDir+"\\", true);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocuments methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSDocuments::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString DocsDir = nt_rec->PrepareDirName(SrcDir);

	ImportOrdering(DocsDir);

	Storage.Create(nt_rec->PrepareStorageName(ID));
	
	for( int i = 0; i < nProps; i++ )
	{
		CMMSDocument* pDoc = (CMMSDocument*)Properties[i];
		pDoc->Compile(Storage, DocsDir, TypedTextStorage);
	}
	
	Storage.Close();

}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSReport methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSReport::Compile(CStorage& Storage, CString& SrcDir, CString& SubStroragePrefix, CStorage& TypedTextStorage)
{
	CString RptDir(SrcDir);
	CString StorName;

	RptDir += "\\";
	RptDir += sID;

	StorName.Format("%s%i", SubStroragePrefix, ID);
	Storage.Create(StorName);
	ImportWorkBook(Storage, RptDir);
	Storage.Close();

	ImportDescriptions(TypedTextStorage, RptDir+"\\", true);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSReportList methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSReportList::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(sID);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString ReportsDir = nt_rec->PrepareDirName(SrcDir);

	ImportOrdering(ReportsDir);

	Storage.Create(nt_rec->StorageName);
	CString ReportStorName = nt_rec->StorageName + "_Number";
	for( int i = 0; i < nProps; i++ )
	{
		CMMSReport* rpt = (CMMSReport*)Properties[i];
		rpt->Compile(Storage, ReportsDir, ReportStorName, TypedTextStorage);
	}
	Storage.Close();
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSJournalister methods  //////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSJournalister::Compile(CStorage& Storage, CString& SrcDir, CStorage& TypedTextStorage)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString JournalDir = SrcDir;
	CMMSObject* Forms = GetProperty(CString("Form"));

	JournalDir += "\\";
	JournalDir += sID;

	ImportDescriptions(TypedTextStorage, JournalDir+"\\", true);
	ImportListForms(Forms, Storage, JournalDir, nt_rec->StorageName);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSJournalisters methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSJournalisters::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString JournalsDir = nt_rec->PrepareDirName(SrcDir);

	ImportOrdering(JournalsDir);

	Storage.Create(nt_rec->StorageName);
	for( int i = 0; i < nProps; i++ )
	{
		CMMSJournalister* pJournal = (CMMSJournalister*)Properties[i];
		pJournal->Compile(Storage, JournalsDir, TypedTextStorage);
	}

	Storage.Close();
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCalcJournal methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCalcJournal::Compile(CStorage& Storage, CString& SrcDir, CStorage& TypedTextStorage)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString JournalDir(SrcDir);
	CMMSObject* Forms = GetProperty(CString("Form"));

	JournalDir += "\\";
	JournalDir += sID;

	ImportDescriptions(TypedTextStorage, JournalDir+"\\", true);
	ImportListForms(Forms, Storage, JournalDir, nt_rec->StorageName);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSCalcJournals methods  //////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCalcJournals::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CStorage TypedTextStorage(Storage, "TypedText");
	CString JournalsDir = nt_rec->PrepareDirName(SrcDir);

	ImportOrdering(JournalsDir);

	Storage.Create(nt_rec->StorageName);
	for( int i = 0; i < nProps; i++ )
	{
		CMMSCalcJournal* pJournal = (CMMSCalcJournal*)Properties[i];
		pJournal->Compile(Storage, JournalsDir, TypedTextStorage);
	}
	Storage.Close();
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCaclAlgorithm methods  /////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCaclAlgorithm::Compile(CStorage& TypedTextStorage, CString& SrcDir)
{
	CNameTableRecord* nt_rec_stor = NameTable.Find(Type);
	CNameTableRecord* nt_rec_file = NameTable.Find(NT_CalcAlgorithm);
	CString AlgDir(SrcDir), FileName;

	AlgDir += "\\";
	AlgDir += sID;

	FileName = nt_rec_file->PrepareFileName(AlgDir);
	if( FileExist(FileName) )
	{
		TypedTextStorage.Create(nt_rec_stor->PrepareStorageName(ID));
		TypedTextStorage.StreamFromFile(nt_rec_file->StorageName, FileName);
		
		CreateTypedTextContainerContents(TypedTextStorage);
		TypedTextStorage.Close();
	}

	ImportDescriptions(TypedTextStorage, AlgDir+"\\", true);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCaclAlgorithms methods  ////////////////////////
///////////////////////////////////////////////////////////////////////
void CMMSCaclAlgorithms::Compile(CStorage& Storage, CString& SrcDir)
{
	CNameTableRecord* nt_rec = NameTable.Find(Type);
	CString AlgsDir = nt_rec->PrepareDirName(SrcDir);

	ImportOrdering(AlgsDir);

	Storage.Open(nt_rec->StorageName); //TypedText already created, we only have to open it
	for( int i = 0; i < nProps; i++ )
	{
		CMMSCaclAlgorithm* pAlg = (CMMSCaclAlgorithm*)Properties[i];
		pAlg->Compile(Storage, AlgsDir);
	}
	Storage.Close();
}

