
#include "UI.h"
#include "CommonFunctions.h"
#include "NameTable.h"
#include "Storage.h"
#include "MMS.h"


CString CUICommand::AsString(bool IsChild)
{
	CString res;
	if( IsChild )
	{
		res.Format("%s, %s", Command, CmdParameter);
	}
	else
	{
		res.Format("%08x: %s, %s, [%02x:%04x] %s, %s", ID, Param1, Hint, HotKeyModifier, HotKey, Command, CmdParameter);
	}
	return res;
}


CUICommands::CUICommands()
{
	NCommands = 0;
	Commands = NULL;
	IsChild = false;
}

CUICommands::~CUICommands()
{
	if( Commands != NULL ) 
	{
		delete[] Commands;
	}
}

bool ReadUIString(CString& String, IStream* pStream)
{
	BYTE len;
	char buf[256];
	ULONG BytesRead;

	pStream->Read(&len, sizeof(len), &BytesRead);
	if( BytesRead != sizeof(len) ) return false;
	pStream->Read(buf, len, &BytesRead);
	if( BytesRead != len ) return false;

	buf[len] = '\0';
	String = buf;
	return true;
}

DWORD CUICommands::Read(CStorage& Storage, CString& UIName)
{
	Storage.OpenStream(CString("Commands"));
	IStream* pStream = Storage.pStream;

	DWORD id;
	DWORD i;
	
	pStream->Read(&id, sizeof(id), NULL);
	if( IsChild )
	{
		ReadUIString(ParentInterface, pStream);
	}
	else
	{
		ReadUIString(Description, pStream);
	}

	pStream->Read(&NoOperMenuPresent, sizeof(NoOperMenuPresent), NULL);
	pStream->Read(&IsDefault, sizeof(IsDefault), NULL);

	if( IsChild )
	{
		ReadUIString(Description, pStream);
	}
	
	pStream->Read(&NCommands, sizeof(NCommands), NULL);

	Commands = new CUICommand[NCommands];
	for( i = 0; i < NCommands; i++ )
	{
		CUICommand& cmd = Commands[i];
		
		if( IsChild )
		{
			if( !ReadUIString(cmd.Command, pStream) )
			{
				Msg(0, "WARN:   %s  %i ,    %i\n", UIName, NCommands, i);
				NCommands = i;
				break;
			}
			if( !ReadUIString(cmd.CmdParameter, pStream) )
			{
				Msg(0, "WARN:   %s  %i ,    %i\n", UIName, NCommands, i);
				NCommands = i;
				break;
			}
		}
		else
		{
			pStream->Read(&cmd.ID, sizeof(cmd.ID), NULL);
			
			ReadUIString(cmd.Param1, pStream);
			ReadUIString(cmd.Hint, pStream);
			
			pStream->Read(&cmd.HotKeyModifier, sizeof(cmd.HotKeyModifier), NULL);
			pStream->Read(&cmd.HotKey, sizeof(cmd.HotKey), NULL);

			ReadUIString(cmd.Command, pStream);
			ReadUIString(cmd.CmdParameter, pStream);
		}
		cmd.CmdParameter.Replace('\n', ':');
	}

	return NCommands;
}

CString CUICommands::AsString()
{
	CString res;

	res += ": ";
	res += StorageName;
	res += "\n";

	if( IsChild )
		res += ": \n";
	else
		res += ": \n";

	res += ": ";
	res += ParentInterface;
	res += "\n";

	res += ": ";
	res += Description;
	res += "\n";

	res += "  : ";
	res += NoOperMenuPresent == 0 ? "" : "";
	res += "\n";

	res += "   : ";
	res += IsDefault == 0 ? "" : "";
	res += "\n";

	res += "{\n";
	for( DWORD i = 0; i < NCommands; i++ )
	{
		res += "\t";
		res += Commands[i].AsString(IsChild);
		res += "\n";
	}
	res += "}";

	return res;
}


CUIMenu::CUIMenu()
{
	Type = MainMenu;
	Parent = NULL;
	Level = 0;
	NItems = 0;
	Items = NULL;
	ID = 0;
}

CUIMenu::~CUIMenu()
{
	if( Items != NULL ) delete[] Items;
}

DWORD CUIMenu::Read(CStorage& Storage, CUIMenu* Parent)
{
	this->Parent = Parent;
	if( Parent != NULL )
	{
		Level = Parent->Level + 1;
	}
	else
	{
		if( !Storage.OpenStream(CString("Page.1")) ) return 0;
	}

	IStream* pStream = Storage.pStream;

	pStream->Read(&ID, sizeof(ID), NULL);

	switch( ID )
	{
		case 0x01      : Type = MainMenu; break;
		case 0xFFFFFFFF: Type = MenuGroup; break;
		case 0x00      : Type = MenuDividor; break;
		default        : Type = MenuItem;
	}

	if( Type == MenuGroup || Type == MenuItem )
	{
		ReadUIString(Caption, pStream);
	}
	else if( Type == MenuDividor )
	{
		Caption = ">----------<";
	}

	if( Type == MainMenu || Type == MenuGroup )
	{
		pStream->Read(&NItems, sizeof(NItems), NULL);
		Items = new CUIMenu[NItems];
		for( DWORD i = 0; i < NItems; i++ )
		{
			Items[i].Read(Storage, this);
		}
		return NItems;
	}

	return 1;
}

CString CUIMenu::AsString()
{
	CString prefix, sID, res;
	
	sID.Format("%08x", ID);
	str_repeate(prefix, "\t", Level);

	res += prefix;
	switch( Type )
	{
		case MainMenu   : res += " :"; break;
		case MenuGroup  : res += ": "; break;
		case MenuItem   : res += ": "+sID+": "; break;
		case MenuDividor: res += ": "; break;
	}
	res += Caption;
	res += "\n";

	if( NItems > 0 )
	{
		res += prefix;
		res += "{\n";
		for( DWORD i = 0; i < NItems; i++ )
		{
			res += Items[i].AsString();
		}
		res += prefix;
		res += "}\n";
	}

	return res;
}

DWORD CUIPanel::Read(CStorage& Storage)
{
	IStream* pStream = Storage.pStream;
	ReadUIString(Caption, pStream);
	pStream->Read(&NButtons, sizeof(NButtons), NULL);
	pStream->Read(&Placement, sizeof(Placement), NULL);
	pStream->Read(&IsVisible, sizeof(IsVisible), NULL);
	pStream->Read(&OnNewLine, sizeof(OnNewLine), NULL);
	return NButtons;
}

CString CUIPanel::AsString()
{
	CString res;
	CString sPlacement, sVisible, sOnNewLine;

	switch( Placement )
	{
		case 0xE81B: sPlacement = ""; break;
		case 0xE81C: sPlacement = ""; break;
		case 0xE81D: sPlacement = ""; break;
		case 0xE81E: sPlacement = ""; break;
		default: 
			sPlacement = "   !!!";
			Msg(0, "ERR:     %s\n", Caption);
	}
	
	if( IsVisible == 0 )
		sVisible = "";
	else
		sVisible = "";

	if( OnNewLine == 0 )
		sOnNewLine = "";
	else
		sOnNewLine = "";

	res.Format(": %s\n{\t: %s\n\t: %s\n\t  : %s\n}\n", 
		Caption, sPlacement, sVisible, sOnNewLine);

	return res;
}

CUIPanels::CUIPanels()
{
	NPanels = 0;
	NButtons = 0;
	Panels = NULL;
	AllButtons = NULL;
	NTextButtons = 0;
	TextButtons = NULL;
	NPictureButtons = 0;
	PictureButtons = NULL;
}

CUIPanels::~CUIPanels()
{
	if( Panels != NULL ) delete[] Panels;
	if( AllButtons != NULL ) delete[] AllButtons;
	if( TextButtons != NULL ) delete[] TextButtons;
	if( PictureButtons != NULL ) delete[] PictureButtons;
}

bool CUIPanels::Read(CStorage& Storage)
{
	CString StreamName = "Page.2";
	long sz = Storage.GetStreamSize(StreamName);
	if( !Storage.OpenStream(StreamName) ) return false;
	Storage.StreamInfo.FullSize = sz;

	IStream* pStream = Storage.pStream;
	DWORD i;

	pStream->Read(&ID, sizeof(ID), NULL);
	pStream->Read(&NPanels, sizeof(NPanels), NULL);
	
	Panels = new CUIPanel[NPanels];
	for( i = 0; i < NPanels; i++ )
	{
		NButtons += Panels[i].Read(Storage);
	}

	AllButtons = new CButton[NButtons];
	for( i = 0; i < NButtons; i++ )
	{
		pStream->Read(&AllButtons[i].ID, sizeof(AllButtons[i].ID), NULL);
	}

	pStream->Read(&NTextButtons, sizeof(NTextButtons), NULL);
	TextButtons = new CButton[NTextButtons];
	for( i = 0; i < NTextButtons; i++ )
	{
		ReadUIString(TextButtons[i].Caption, pStream);
		pStream->Read(&TextButtons[i].ID, sizeof(TextButtons[i].ID), NULL);
	}

	pStream->Read(&NPictureButtons, sizeof(NPictureButtons), NULL);
	PictureButtons = new CButton[NPictureButtons];
	for( i = 0; i < NPictureButtons; i++ )
	{
		pStream->Read(&PictureButtons[i].ID, sizeof(PictureButtons[i].ID), NULL);
	}

	LARGE_INTEGER SeekZero = {0, 0};
	ULARGE_INTEGER pos;
	pStream->Seek(SeekZero, STREAM_SEEK_CUR, &pos);
	BMP_offset = pos.LowPart;

	return true;
}

void CUIPanels::ExtractBMP(CString& Dir, CStorage& Storage)
{
	LARGE_INTEGER offset = {BMP_offset, 0};
	CString StreamName = "Page.2";
	if( !Storage.OpenStream(StreamName) ) return;
	Storage.pStream->Seek(offset, STREAM_SEEK_SET, NULL);
	Storage.CopyToFile(Dir+"\\.bmp", true);
}

CString CButton::AsString(bool WithCaption)
{
	CString res;
	if( WithCaption )
	{
		res.Format("\t%08x: %s\n", ID, Caption);
	}
	else
	{
		res.Format("\t%08x\n", ID);
	}

	return res;
}

CString CUIPanels::AsString()
{
	CString res;
	DWORD i;

	for( i = 0; i < NPanels; i++ )
	{
		res += Panels[i].AsString();
	}

	res += ":\n{\n";
	for( i = 0; i < NButtons; i++ )
	{
		res += AllButtons[i].AsString(false);
	}
	res += "}\n";

	res += " :\n{\n";
	for( i = 0; i < NTextButtons; i++ )
	{
		res += TextButtons[i].AsString(true);
	}
	res += "}\n";

	res += "  :\n{\n";
	for( i = 0; i < NPictureButtons; i++ )
	{
		res += PictureButtons[i].AsString(false);
	}
	res += "}\n";

	return res;
}

CUserInterfaces::~CUserInterfaces()
{
	int i = GetSize();
	while( --i >= 0 )
	{
		delete GetAt(i);
	}
}

CUserInterface* CUserInterfaces::FindInterface(CString& Name)
{
	int i = GetSize() - 1;
	while( i >= 0 )
	{
		if( GetAt(i)->Name == Name ) return GetAt(i);
		CUserInterface* Interface = GetAt(i)->Children.FindInterface(Name);
		if( Interface != NULL ) return Interface;
		i--;
	}
	return NULL;
}

void CUserInterfaces::AddInterface(CUserInterface* Interface)
{
	if( Interface->Commands.IsChild )
	{
		CUserInterface* Parent = FindInterface(Interface->Commands.ParentInterface);
		Parent->Children.Add(Interface);
		Interface->Parent = Parent;
	}
	else
	{
		Add(Interface);
	}
}

void CUserInterfaces::Extract(CString& Dir, CStorage& Storage)
{
	for( int i = 0; i < GetSize(); i++ )
	{
		GetAt(i)->Extract(Dir, Storage);
	}
}

void CUserInterface::Read(CString& Dir, CStorage& Storage, CMMSObject* Container)
{
	StorageName = Container->GetPropByNum(0)->sID;
	Name = Container->GetPropByNum(1)->sID;
	Msg(2, " %s\n", Name);

	Storage.Open(StorageName);
	
	if( Container->sID == "UsersInterfaceType" )
	{
		Commands.IsChild = false;
	}
	else if( Container->sID == "SubUsersInterfaceType" )
	{
		Commands.IsChild = true;
	}

	Commands.StorageName = StorageName;
	Commands.Read(Storage, Name);
	if( !Commands.IsChild )
	{
		Menu.Read(Storage, NULL);
		Panels.Read(Storage);
	}

	Storage.Close();
}

void CUserInterface::Extract(CString& Dir, CStorage& Storage)
{
	CString UIDir = Dir + "\\" + Name;
	CreateDirectory(UIDir);

	CopyStringToFile(Commands.AsString(), UIDir+"\\Commands.txt");
	if( !Commands.IsChild )
	{
		CopyStringToFile(Menu.AsString(), UIDir+"\\Menu.txt");
		CopyStringToFile(Panels.AsString(), UIDir+"\\.txt");

		Storage.Open(StorageName);
		Panels.ExtractBMP(UIDir, Storage);
		Storage.Close();
	}

	Children.Extract(UIDir, Storage);
}

bool DecompileUI(CString& Dir, CStorage& Storage)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_UserInterface);
	CString DestDir = nt_rec->PrepareDirName(Dir);
	CStorage UIStorage(Storage);
	CMMSNodeList* CC;
	CUserInterfaces UserInterfaces;

	if( !UIStorage.Open(nt_rec->PrepareStorageName()) )
	{
		return false;
	}

	CC = (CMMSNodeList*)UIStorage.ParseStream(CString("Container.Contents"));
	if( CC == NULL ) return false;

	CreateDirectory(DestDir);

	POSITION pos = CC->GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* UI = CC->GetNext(pos);
		CUserInterface* Interface = new CUserInterface;
		Interface->Read(DestDir, UIStorage, UI);
		UserInterfaces.AddInterface(Interface);
	}

	UserInterfaces.Extract(DestDir, UIStorage);
	
	return true;
}