// TaskCompile.cpp: implementation of the CTaskCompile class.
//////////////////////////////////////////////////////////////////////

#include <io.h>
#include "Task.h"
#include "Container.h"
#include "CommonFunctions.h"
#include "Storage.h"
#include "MMS.h"
#include "NameTable.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define BUFFER_SIZE 256

extern CTaskParameters TaskParameters;
extern CNameTable NameTable;

//////////////////////////////////////////////////////////////////////
void AddFile(FILE* DestFile, CString& FName, bool AddComma = true)
{
	char buff[BUFFER_SIZE];
	long BytesRead;
	FILE *SrcFile = fopen((LPCSTR)FName, "rb");

	if( SrcFile == NULL )
	{
		Msg(0, "ERR: Can't read file '%s' - %s\n", (LPCSTR)FName, strerror(errno));
		exit(1);
	}

	if( ftell(DestFile) > 0 && AddComma )
	{
		fputs(",\r\n", DestFile);
	}

	do
	{
		BytesRead = fread(buff, 1, sizeof(buff), SrcFile);
		fwrite(buff, 1, BytesRead, DestFile);
	} while( BytesRead == sizeof(buff) );

	fclose(SrcFile);
}

void AddDir(FILE* DestFile, CString& DirName, LPCSTR sID)
{
	CNameTableRecord* nt_rec = NameTable.Find(NT_MetadataPart);
	CString FName = nt_rec->PrepareFileName(DirName);

	fputs(",\r\n{\"", DestFile);
	fputs(sID, DestFile);
	fputs("\"\r\n", DestFile);
	if( FileExist(FName) ) AddFile(DestFile, FName);

	struct _finddata_t find_data;
	long hFind;

	hFind = _findfirst(DirName+"\\*", &find_data);
	if( hFind != -1 )
	{
		do
		{
			if (0 == strcmp(find_data.name, ".")) continue;
			if (0 == strcmp(find_data.name, "..")) continue;
			if (0 == stricmp(find_data.name, "CVS")) continue;
			if( (find_data.attrib & _A_SUBDIR) == 0 ) continue;

			FName.Format("%s\\%s\\%s.%s", DirName, find_data.name, nt_rec->FileName, nt_rec->Ext);
			if( !FileExist(FName) ) continue;

			AddFile(DestFile, FName);
		} while( _findnext(hFind, &find_data) == 0 );
		_findclose(hFind);
	}

	fputs("\r\n}", DestFile);
}


FILE* AssembleMMS()
{
	FILE *mmsFile;

	mmsFile = tmpfile();
	if( mmsFile == NULL )
	{
		Msg(0, "ERR: Can't create temporary file\n");
		exit(1);
	}

	CString Dir = TaskParameters.DirectoryName;
	
	fputs("{\r\n", mmsFile);
	

	bool AddComma = false;
	for( int i = 0; i < NameTable.GetNRecords(); i++ )
	{
		CNameTableRecord *nt_entry = NameTable[i];

		if( nt_entry->sID.GetLength() == 0 ) continue; //    

		if( nt_entry->FileName.GetLength() != 0 )
			AddFile(mmsFile, nt_entry->PrepareFileName(Dir), AddComma);
		else
			AddDir(mmsFile, Dir+"\\"+nt_entry->Dir, nt_entry->sID);

		AddComma = true;
	}

	fputs("\r\n}", mmsFile);

	return mmsFile;
}

void ImportMMS(CStorage& Storage, FILE* mmsFile, CMMSObject* MMSObject)
{
	FILE *mmsFile_;

	mmsFile_ = tmpfile();
	fseek(mmsFile_, 0, SEEK_SET);

	CMMSDescr* Descr = (CMMSDescr*)MMSObject->GetPropByNum(0);
	Descr->MD_ver = 10009;

	fputs(MMSObject->AsString(-100), mmsFile_);

	Storage.Create(CString("Metadata"));
	Storage.CreateStream(CString("Main MetaData Stream"));
	Storage.FromFile(mmsFile_, "Main MetaData Stream");
	Storage.Close();

	fclose(mmsFile_);
}

void CTaskCompile::Action()
{
	CTime Time = CTime::GetCurrentTime();
	Msg(0, "\nCompilation started: %s\n", Time.Format("%Y.%m.%d %H:%M:%S"));

	if( !DirectoryExist(TaskParameters.DirectoryName) )
	{
		Msg(0, "ERR: Source directory '%s' not exist\n", TaskParameters.DirectoryName);
		return;
	}

	FILE *mmsFile = AssembleMMS();
	CMMSObject* MMSObject;
	rewind(mmsFile);
	MMS_InitParser();
	int RetCode = MMS_parse(mmsFile, &MMSObject);

	CStorage Storage(TaskParameters.CompoundFileName);
	if( Storage.GetStorage() != NULL )
	{
		CoInitialize(NULL);
		ImportMMS(Storage, mmsFile, MMSObject);
		MMSObject->Compile(Storage, TaskParameters.DirectoryName);
		CoUninitialize();
	}
	fclose(mmsFile);

	delete MMSObject;

	CTimeSpan CompilationTime = CTime::GetCurrentTime() - Time;
	CString FileName(TaskParameters.CompoundFileName);
	CString DirName(TaskParameters.DirectoryName);
	FileName.AnsiToOem();
	DirName.AnsiToOem();

	Msg(0, "Dir '%s' compiled into '%s'\n", TaskParameters.DirectoryName, TaskParameters.CompoundFileName);
	Msg(0, "Compilation takes: %s\n", CompilationTime.Format("%H:%M:%S"));
}
