
#include <io.h>
#include <locale.h>
#include <direct.h>
#include "DLL\GComp.h"
#include "Declarations.h"
#include "CommonFunctions.h"
#include "Lexer.h"
#include "MMS.h"
#include "Storage.h"
#include "NameTable.h"
#include "Ordering.h"
#include "MMS_Filter.h"

CGCompParameters TaskParameters;
CNameTable NameTable;
CMMSFilter MMS_Filter;
CMMS* MainMetadataStream = NULL;


LPCSTR GComp_GetVersion()
{
	return "2.1.3.beta";
}


void GComp_SetDefaults(CGCompParameters* Parameters) 
{
	Parameters->CompoundFileName = NULL;
	Parameters->DirectoryName = NULL;

	Parameters->CompoundType = by_extension;
	Parameters->Verbose = 0;
	Parameters->LogType = LogType_stdout;
	Parameters->hwndLog = NULL;
	Parameters->NoOrdering = FALSE;
	Parameters->NoProfiles = FALSE;
	Parameters->NoEmptyMxl = FALSE;
	Parameters->TruncateMMS = FALSE;
	Parameters->NotParseForms = FALSE;
	Parameters->NoDefaultsInDialogs = FALSE;
	Parameters->Language = lang_Rus;
	Parameters->SaveVersion = TRUE;
	Parameters->MetadataBySubfolders = FALSE;
	Parameters->MetadataPath = NULL;
	Parameters->SaveWholeMMS = FALSE;
};

BOOL YesNoToBool(CString& token, CString& value)
{
	if( value.CompareNoCase("yes")   == 0 ) return true;
	if( value.CompareNoCase("y")     == 0 ) return true;
	if( value.CollateNoCase("")    == 0 ) return true;
	if( value.CompareNoCase("no")    == 0 ) return false;
	if( value.CompareNoCase("n")     == 0 ) return false;
	if( value.CollateNoCase("")   == 0 ) return false;

	Msg(0, "ERR:   '%s'  %s  gcomp.ini", value, token);
	throw NULL;
}

TGComp_Languages ReadLanguage(CString& token, CString& value)
{
	if( value.CompareNoCase("rus") == 0 )
		return lang_Rus;
	else if( value.CompareNoCase("eng") == 0 )
		return lang_Eng;
	else if( value.CompareNoCase("translit") == 0 )
		return lang_Translit;
	else
	{
		Msg(0, "ERR:   '%s'  gcomp.ini", value);
		throw NULL;
	}
}

void GComp_ReadSettings(CGCompParameters* Parameters)
{
	CString ini_name;
	ini_name.Format("%s\\gcomp.ini", Parameters->DirectoryName);
	if( !FileExist(ini_name) ) return;

	CLexer ini_file(ini_name);
	CString token, value;
	while( ini_file.GetToken(token, "=: \t\r\n") )
	{
		ini_file.GetRest(value);
		if( stricmp(token, "no-ordering") == 0 )
			Parameters->NoOrdering = YesNoToBool(token, value);
		
		else if( stricmp(token, "no-profiles") == 0 )
			Parameters->NoProfiles = YesNoToBool(token, value);
		
		else if( stricmp(token, "no-empty-mxl") == 0 )
			Parameters->NoEmptyMxl = YesNoToBool(token, value);
		
		else if( stricmp(token, "no-parse-forms") == 0 )
			Parameters->NotParseForms = YesNoToBool(token, value);
		
		else if( stricmp(token, "no-dialog-defaults") == 0 )
			Parameters->NoDefaultsInDialogs = YesNoToBool(token, value);

		else if( stricmp(token, "language") == 0 )
			Parameters->Language = ReadLanguage(token, value);

		else if( stricmp(token, "more-subfolders") == 0 )
			Parameters->MetadataBySubfolders = YesNoToBool(token, value);

		else if( stricmp(token, "save-whole-mms") == 0 )
			Parameters->SaveWholeMMS = YesNoToBool(token, value);

		else
		{
			Msg(0, "ERR:   '%s'  gcomp.ini", token);
			throw NULL;
		}
	}
}


inline char* BoolToYesNo(BOOL value)
{
	return value ? "Yes" : "No";
}

char* LanguageToStr(TGComp_Languages lang)
{
	switch( lang )
	{
	case lang_Rus:
		return "Rus";
		break;
	case lang_Eng:
		return "Eng";
		break;
	case lang_Translit:
		return "Translit";
		break;
	}
	Msg(0, "ERR:   %i", (int)lang);
	throw NULL;
}

void GComp_SaveSettings(CGCompParameters* Parameters)
{
	CString ini_name;
	ini_name.Format("%s\\gcomp.ini", Parameters->DirectoryName);
	if( FileExist(ini_name) ) return;

	FILE* ini_file = fopen(ini_name, "wt");
	fprintf(ini_file, "no-ordering = %s\n",        BoolToYesNo(Parameters->NoOrdering));
	fprintf(ini_file, "no-profiles = %s\n",        BoolToYesNo(Parameters->NoProfiles));
	fprintf(ini_file, "no-empty-mxl = %s\n",       BoolToYesNo(Parameters->NoEmptyMxl));
	fprintf(ini_file, "no-parse-forms = %s\n",     BoolToYesNo(Parameters->NotParseForms));
	fprintf(ini_file, "no-dialog-defaults = %s\n", BoolToYesNo(Parameters->NoDefaultsInDialogs));
	fprintf(ini_file, "language = %s\n",           LanguageToStr(Parameters->Language));
	fprintf(ini_file, "more-subfolders = %s\n",    BoolToYesNo(Parameters->MetadataBySubfolders));
	fprintf(ini_file, "save-whole-mms = %s\n",     BoolToYesNo(Parameters->SaveWholeMMS));

	Msg(1, "NEW: %s", ini_name);
}

void InitializeNameTable()
{
	if( lang_Eng == TaskParameters.Language )
		NameTable.EngConstructor();
	else
		NameTable.RusConstructor();
}

BOOL GComp_Decompile(CGCompParameters* Parameters)
{
	TaskParameters = *Parameters;
	char* locale = setlocale( LC_ALL, ".ACP");
	CString csDirectoryName = TaskParameters.DirectoryName;

	try
	{
		InitializeNameTable();

		MMS_Filter.Init(TaskParameters.MetadataPath);

		CStorage Storage;
		Storage.OpenCompound(CString(TaskParameters.CompoundFileName));
		CStorage TypedTextStorage(Storage, "TypedText");

		if( Storage.GetStorage() == NULL || TypedTextStorage.GetStorage() == NULL )
			return FALSE;
		
		CoInitialize(NULL); //need it for unzip modules

		CNameTableRecord* nt_rec = NameTable.Find(NT_MainMetadataStream);
		if( external_report == TaskParameters.CompoundType )
		{
			MainMetadataStream = (CMMS*)Storage.ParseStream(nt_rec->StorageName, false);
		}
		else
		{
			Storage.Open(CString("Metadata"));
			MainMetadataStream = (CMMS*)Storage.ParseStream(nt_rec->StorageName, false);
			Storage.Close();
		}

		if( MainMetadataStream != NULL )
		{
			CreateDirectory(csDirectoryName); //create directory for decompyle to

			if( TaskParameters.SaveVersion )
			{
				CString Version = GComp_GetVersion();
				CString FName = csDirectoryName + "\\gcomp.ver";
				Storage.CopyStringToFile(Version, FName);
			}

			MainMetadataStream->Decompile(csDirectoryName, Storage, TypedTextStorage);
			delete MainMetadataStream;

			if( MMS_Filter.IsEmpty() )
				Storage.PruneDirectory(csDirectoryName);
		}

		CoUninitialize(); //need it for unzip modules
	}
	/*   -  
	catch(CGCompError* e)
	{
		if( MMS != NULL ) delete MMS;
		e->LogError();
		return false;
	}
	*/
	catch(...)
	{
		if( MainMetadataStream != NULL ) delete MainMetadataStream;
		return FALSE;
	}

	if( MainMetadataStream == NULL )
		return FALSE;
	else
		return TRUE;
}

///////////////////////////////////////////////////////////////////////
//////  Compilation functions  ////////////////////////////////////////
///////////////////////////////////////////////////////////////////////
FILE* AssembleMMS();
extern int MMS_debug;

void MakeCopyOfMMS(FILE* mmsFile)
{
	FILE *f = fopen("MMS.mdp", "wb");
	char cwd[MAX_PATH];
	_getcwd(cwd, sizeof(cwd));
	Msg(0, "   MainMetadataStream  : %s\\MMS.mdp", cwd);
	CopyFile(f, mmsFile);
	fclose(f);
}

BOOL GComp_Compile(CGCompParameters* Parameters)
{
	TaskParameters = *Parameters;
	char* locale = setlocale( LC_ALL, ".ACP");
	FILE *mmsFile = NULL;
	CString csDirectoryName = TaskParameters.DirectoryName;

	try
	{
		InitializeNameTable();
		MMS_Filter.Init(TaskParameters.MetadataPath);

		if( !DirectoryExist(csDirectoryName) )
		{
			Msg(0, "ERR: Source directory '%s' not exist", TaskParameters.DirectoryName);
			return FALSE;
		}
		CStorage Storage;

		if( MMS_Filter.IsEmpty() )
		{
			if( !Storage.CreateCompound(CString(TaskParameters.CompoundFileName)) )
				return FALSE;

			mmsFile = AssembleMMS();
			rewind(mmsFile);

			MMS_InitParser(csDirectoryName+"\\MainMetadataStream");
			if( TaskParameters.Verbose >= 3) MMS_debug = 1;
			int RetCode = MMS_parse(mmsFile, (CMMSObject**)&MainMetadataStream);

			if( RetCode != 0 || MainMetadataStream == NULL )
				MakeCopyOfMMS(mmsFile);
			fclose(mmsFile);
			mmsFile = NULL;
		}
		else
		{
			//     compound
			CNameTableRecord* nt_rec = NameTable.Find(NT_MainMetadataStream);
			Storage.OpenCompound(CString(TaskParameters.CompoundFileName));
			Storage.Open(CString("Metadata"));
			MainMetadataStream = (CMMS*)Storage.ParseStream(nt_rec->StorageName, false);
			Storage.Close();
		}

		if( MainMetadataStream == NULL )
			return FALSE;

		CoInitialize(NULL);
		CStorage TypedTextStorage(Storage);
		if( TaskParameters.CompoundType != external_report )
			TypedTextStorage.Create(CString("TypedText"));
		MainMetadataStream->Compile(Storage, TypedTextStorage, csDirectoryName);
		CoUninitialize();
	}
	catch(...)
	{
		if( mmsFile != NULL ) fclose(mmsFile);
		if( MainMetadataStream != NULL ) delete MainMetadataStream;
		return false;
	}

	delete MainMetadataStream;
	return TRUE;
}


bool NameToPath(CString Name, CString& Path);
BOOL GComp_MakeNamePath(LPCTSTR srcDir, BOOL translite, LPTSTR objName, int buffsize)
{
	CString Name = objName, Path = srcDir;

	TaskParameters.Language = lang_Rus;
	InitializeNameTable();

	BOOL res = NameToPath(Name, Path);
	Msg(0, "Name: %s, Path: %s", Name, Path);
	return res;
}
