
#include "MMS.h"
#include "NameTable.h"
#include "CommonFunctions.h"
#include "Strings.h"


////////////////////////////////////////////////////////////
//     
int PropsCompareByOrder( const void *arg1, const void *arg2 )
{
	CMMSObject* obj1 = *(PCMMSObject*)arg1;
	CMMSObject* obj2 = *(PCMMSObject*)arg2;

	if( obj1->Order < obj2->Order ) return -1;
	if( obj1->Order > obj2->Order ) return 1;
	return 0;
}

//       ( ID)
int __cdecl ObjNameCache_compare_id(const void *elem1, const void *elem2)
{
	int id1 = *(int*)elem1;
	int id2 = *(int*)elem2;
	if( id1 < id2 ) return -1;
	if( id1 > id2 ) return 1;
	return 0;
}

//       (  ID)
int __cdecl ObjNameCache_compare_str(const void *elem1, const void *elem2)
{
	TObjNameCacheRecord* rec1 = (TObjNameCacheRecord*)elem1;
	TObjNameCacheRecord* rec2 = (TObjNameCacheRecord*)elem2;
	//return rec1->Name.Collate(rec2->Name);
	return strcmp(rec1->Name, rec2->Name);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSObject methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSObject::CMMSObject(CString& sID, CString& ID)
{
	Type = MMS_Object;
	ID_Type = IDT_BothIDs;
	this->sID = sID;
	this->ID = atoi(ID);
	Used = false;
}

CMMSObject::~CMMSObject()
{
	DestroyObjects();
}

void CMMSObject::DestroyObjects()
{
	int nProps = GetNProps();
	while( nProps > 0 )
	{
		CMMSObject* obj = GetPropByNum(--nProps);
		if( obj != NULL ) delete obj;
	}
}


CMMSObject* CMMSObject::GetProperty(int ID, bool recursive, CMMSNodeList* pList)
{
	int nProps = GetNProps();
	int i;
	for( i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type != MMS_Property && obj->ID == ID )
		{
			if( pList != NULL ) pList->AddTail(obj);
			return obj;
		}
	}

	if( !recursive )
		return NULL;

	for( i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type == MMS_Property ) continue;
		CMMSObject* found = obj->GetProperty(ID, true, pList);
		if( found != NULL )
		{
			if( pList != NULL ) pList->AddHead(obj);
			return found;
		}
	}

	return NULL;
}

CMMSObject* CMMSObject::GetProperty(CString& sID, bool recursive, int* pIndex)
{
	int nProps = GetNProps();
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type != MMS_Property && obj->sID == sID )
		{
			if( pIndex != NULL ) *pIndex = i;
			return obj;
		}
	}

	if( recursive )
	{
		for( i = 0; i < nProps; i++ )
		{
			CMMSObject* obj = GetPropByNum(i);
			if( obj->Type == MMS_Property ) continue;
			CMMSObject* found = obj->GetProperty(sID, true, pIndex);
			if( found != NULL )
			{
				if( pIndex != NULL ) *pIndex = i;
				return found;
			}
		}
	}

	if( pIndex != NULL ) *pIndex = -1;
	return NULL;
}


void CMMSObject::BuildObjNameCache(CObjNameCache& Cache, CString& NamePrefix, int level)
{
	int nProps = GetNProps();
	int i;
	TObjNameCacheRecord new_entry;

	for( i = 0; i < nProps; i++ )
	{
		CMMSObject* obj = GetPropByNum(i);
		if( obj->Type == MMS_Property ) continue;
		
		CNameTableRecord* nt_rec = NameTable.Find(obj->Type);
		CString* CurObjName = NULL;

		new_entry.ID = obj->ID;
		new_entry.Name = NamePrefix;

		if( nt_rec == NULL )
			CurObjName = &(obj->sID);
		else if( nt_rec->Type1C.IsEmpty() )
			CurObjName = NULL;
		else if( nt_rec->Type1C != "sID" )
			CurObjName = &(nt_rec->Type1C);
		else if( obj->Type == MMS_Account ) //   .       
			CurObjName = &( ((CMMSAccount*)obj)->Code);
		else
			CurObjName = &(obj->sID);

		if( CurObjName != NULL && !CurObjName->IsEmpty() )
		{
			if( !NamePrefix.IsEmpty() ) new_entry.Name += ".";
			new_entry.Name += *CurObjName;
		}

		if( obj->ID != 0 && CurObjName != NULL && !CurObjName->IsEmpty() )
			Cache.Add(new_entry);

		obj->BuildObjNameCache(Cache, new_entry.Name, level+1);
	}
}




void CMMSObject::MarkUsed(int ID)
{
	CMMSNodeList List;
	List.DestroyObjects = false;
	if( GetProperty(ID, true, &List) == NULL ) return;

	POSITION pos = List.GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* SubNode = List.GetNext(pos);
		SubNode->Used = true;
	}
	
}

void CMMSObject::DeleteUnused()
{
	int i = GetNProps() - 1;
	while( i >= 0 )
	{
		CMMSObject* prop = GetPropByNum(i);
		if( prop->Used )
			prop->DeleteUnused();
		else if( prop->Type != MMS_Property )
			DelProperty(i);

		i--;
	}
}

CString CMMSObject::HdrAsString()
{
	CString hdr;

	if( ID_Type == IDT_NoID ) return "";

	if( Type == MMS_Property ) 
	{
		if( ID_Type == IDT_OnlyNumeric )
			hdr.Format("\"%i\"", ID);
		else
			hdr.Format("\"%s\"", sID);
	}
	else
	{
		if( ID_Type == IDT_OnlyNumeric )
			hdr.Format("\"%i\"", ID);
		else if( ID_Type == IDT_OnlyTextual )
			hdr.Format("\"%s\"", sID);
		else
			hdr.Format("\"%i\",\"%s\"", ID, sID);
	}

	return hdr;
}

bool CMMSObject::PropsAsString(CString& dest, int Depth)
{
	int NProps = GetNProps();
	
	if( NProps == 0 ) return false;

	CString comma(",");
	CString offset("\r\n");
	str_repeate(offset, "\t", Depth+1);
	bool HasObjects = false;
	for( int i = 0; i < NProps; i++ )
	{
		CMMSObject* prop = GetPropByNum(i);
		if( i > 0 ) dest += comma;
		if( prop->Type != MMS_Property )
		{
			dest += offset;
			HasObjects = true;
		}
		dest += GetPropByNum(i)->AsString(Depth+1);
	}

	return HasObjects;
}

CString CMMSObject::AsString(int Depth)
{
	if( Type == MMS_Property ) return CString("\"") + sID + "\"";

	CString str("{"), brace_offset("\r\n");
	str_repeate(brace_offset, "\t", Depth);

	str += HdrAsString();

	bool HasObjects = false;
	if( GetNProps() > 0 )
	{
		if( ID_Type != IDT_NoID ) str += ",";
		if( PropsAsString(str, Depth) ) HasObjects = true;
	}
	
	if( HasObjects && Depth >= 0 ) str += brace_offset;
	str += "}";

	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSObjectWithProps methods  ///////////////////////
///////////////////////////////////////////////////////////////////////
CMMSObjectWithProps::CMMSObjectWithProps(CMMSNodeList *pList)
{
	SetProperties(pList);
}

CMMSObjectWithProps::CMMSObjectWithProps(CString& sID, CString& ID, CMMSNodeList *pList)
:CMMSObject(sID, ID)
{
	SetProperties(pList);
}

CMMSObjectWithProps::~CMMSObjectWithProps()
{
	DestroyObjects();
}

void CMMSObjectWithProps::SetProperties(CMMSNodeList *pList)
{
	if( pList == NULL )
	{
		nProps = 0;
		return;
	}

	pList->DestroyObjects = false;

	Properties.SetSize(pList->GetCount());
	POSITION pos = pList->GetHeadPosition();
	nProps = 0;
	while( pos != NULL )
	{
		CMMSObject* obj = pList->GetNext(pos);
		obj->Order = nProps;
		Properties.SetAt(nProps, obj);
		nProps++;
	}
}

CMMSObject* CMMSObjectWithProps::AddProperty(CMMSObject* Property)
{
	Properties.Add(Property);
	nProps++;
	return Property;
}

bool CMMSObjectWithProps::DelProperty(int i) 
{
	if( nProps == 0 ) return false;
	CMMSObject* obj = Properties.GetAt(i);
	Properties.RemoveAt(i);
	delete obj;
	nProps--;
	return true;
}


void CMMSObjectWithProps::SortByOrder()
{
	qsort(Properties.GetData(), nProps, sizeof(CMMSObject*), PropsCompareByOrder);
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSBasicObject methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
CString CMMSBasicObject::GetObjectDir(CString& ParentDir)
{
	CString Result(ParentDir);
	Result += "\\";
	Result += Translit(sID);
	return Result;
}

CMMSBasicObject::CMMSBasicObject(CString& sID, CString& ID, CString& Comment, CString& Synonym, CMMSNodeList *pList)
:CMMSObjectWithProps(sID, ID, pList)
{
	Type = MMS_Object;
	this->Comment = Comment;
	this->Synonym = Synonym;
}


CString CMMSBasicObject::HdrAsString()
{
	CString str;
	str.Format("\"%i\",\"%s\",\"%s\",\"%s\"", ID, sID, Comment, Synonym);
	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMS methods  //////////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMS::CMMS(CMMSNodeList *pList)
:CMMSObjectWithProps(pList)
{
}

CString CMMS::AsString(int Depth)
{
	CString str("{");
	PropsAsString(str, Depth);
	if( Depth >= 0 )
		str += "\r\n}\r\n";
	else
		str += "}\r\n";
	return str;
}


CString& CMMS::ObjectIDtoName(int ID)
{
	if( ID == 0 ) return EmptyObjectName;

	TObjNameCacheRecord* cache_entry = 
		(TObjNameCacheRecord*) bsearch(&ID, ObjNameCache.GetData(), 
		ObjNameCache.GetSize(), sizeof(TObjNameCacheRecord),
		ObjNameCache_compare_id);
	if( cache_entry != NULL )
		return cache_entry->Name;
	else
	{
		//Msg(0, "ERR:      %i", ID);
		return EmptyObjectName;
		//throw NULL;
	}
}

int CMMS::ObjectNameToID(CString& Name)
{
	if( Name.IsEmpty() ) return 0;

	TObjNameCacheRecord record;
	record.Name = Name;

	TObjNameCacheRecord* cache_entry = 
		(TObjNameCacheRecord*) bsearch(&record, ObjNameCache.GetData(), 
		ObjNameCache.GetSize(), sizeof(TObjNameCacheRecord),
		ObjNameCache_compare_str);
	if( cache_entry != NULL )
		return cache_entry->ID;
	else
	{
		Msg(0, "ERR:      %s", Name);
		return 0;
		//throw NULL;
	}
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDescr methods  /////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSDescr::CMMSDescr(CString& sID, CString& FirstAvailableID, CString& MD_ver, CString& UnknownID)
:CMMSMetadataPart(MMS_MainDataContDef)
{
	ID = 0;
	this->sID = sID;
	this->FirstAvailableID = atoi((LPCSTR)FirstAvailableID);
	this->MD_ver = atoi((LPCSTR)MD_ver);
	this->UnknownID = atoi((LPCSTR)UnknownID);
}

CString CMMSDescr::AsString(int Depth)
{
	CString str;
	str.Format("{\"%s\",\"%i\",\"%i\",\"%i\"}", sID, FirstAvailableID, MD_ver, UnknownID);
	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSSubCnt methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSSbCnt::CMMSSbCnt(CString& sID, CString& ID, CString& Comment, CString& Synonym, CMMSNodeList *pList)
:CMMSBasicObject(sID, ID, Comment, Synonym, pList)
{
	Type = MMS_SbCnt;
	GetPropByNum(14)->Type = MMS_SbCntParams;
	GetPropByNum(15)->Type = NT_SubcontoListForm;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSSbCnts methods  ////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSSbCnts::CMMSSbCnts(CString sID, CMMSNodeList* pDocList)
:CMMSObjectWithProps(sID, CString())
{
	Type = MMS_SbCnts;
	ID_Type = CMMSObject::IDT_OnlyTextual;
	SetProperties(pDocList);
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocuments methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSDocument::CMMSDocument(CString& sID, CString& ID, 
						   CString& Comment, CString& Synonym, 
						   CMMSNodeList *pList)
:CMMSBasicObject(sID, ID, Comment, Synonym, pList)
{
	Type = MMS_Document;
	GetPropByNum(18)->Type = MMS_DocHeadFields;
	GetPropByNum(19)->Type = MMS_DocTableFields;
}

CMMSDocuments::CMMSDocuments(CString sID, CMMSNodeList* pDocList)
:CMMSObjectWithProps(sID, CString(""), pDocList)
{
	Type = MMS_Documents;
	ID_Type = CMMSObject::IDT_OnlyTextual;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSDocStream methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSDocStream::CMMSDocStream(CString& sID, CString& ID, 
							 CString& Comment, CString& Synonym, 
							 CString& Param1, CString& Param2, 
							 CMMSObject* Registers, CMMSObject* Documents)
:CMMSObject(sID, ID)
{
	Type = MMS_DocumentStream;
	this->Comment = Comment;
	this->Synonym = Synonym;
	this->Param1 = Param1;
	this->Param2 = Param2;
	this->Registers = Registers;
	this->Documents = Documents;
}

CMMSDocStream::~CMMSDocStream()
{
	if( Registers != NULL ) delete Registers;
	if( Documents != NULL ) delete Documents;
}

CString CMMSDocStream::AsString(int Depth)
{
	CString str, comma(","), offset;
	str_repeate(offset, "\t", Depth+1);
	str.Format("{\"%i\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\r\n%s%s,\r\n%s%s}", 
		ID, sID, 
		Comment, Synonym, Param1, Param2,
		offset, Registers->AsString(Depth+1),
		offset, Documents->AsString(Depth+1));

	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSJournalister methods  //////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSJournalister::CMMSJournalister(CString& sID, CString& ID, CString& Comment, CString& Synonym, CMMSNodeList* pList)
:CMMSBasicObject(sID, ID, Comment, Synonym, pList)
{
	Type = MMS_Journalister;
	GetPropByNum(6)->Type = MMS_JournalFlds;
	GetPropByNum(7)->Type = NT_ListForm;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSCalcJournal methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSCalcJournal::CMMSCalcJournal(CString& sID, CString& ID,
								 CString& Comment, CString& Synonym, 
								 CMMSNodeList* pList)
:CMMSBasicObject(sID, ID, Comment, Synonym, pList)
{
	Type = MMS_CalcJournal;
	GetPropByNum(3)->Type = MMS_JournalFlds;
	GetPropByNum(8)->Type = NT_ListForm;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSRegisters methods  /////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSRegisters::CMMSRegisters(MMSObjectType Type, CString& sID, CMMSNodeList* pPropsList)
:CMMSMetadataPart(Type, sID, pPropsList)
{
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* Register = GetPropByNum(i);
		Register->Type = MMS_Register;
		Register->GetPropByNum(6)->Type = MMS_RegisterProps;
		Register->GetPropByNum(7)->Type = MMS_RegisterFigures;
		Register->GetPropByNum(8)->Type = MMS_DocumentFlds;
	}
}


///////////////////////////////////////////////////////////////////////
CMMSEnums::CMMSEnums(MMSObjectType Type, CString& sID, CMMSNodeList* pPropsList)
:CMMSMetadataPart(Type, sID, pPropsList)
{
	for( int i = 0; i < nProps; i++ )
	{
		CMMSObject* Enum = GetPropByNum(i);
		Enum->Type = MMS_Enum;
		Enum->GetPropByNum(2)->Type = MMS_EnumVal;
	}
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSBuh methods  ///////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSBuh::CMMSBuh(CString sID, CString& BuhID)
:CMMSObjectWithProps(sID, BuhID, NULL)
{
	Type = MMS_Buh;
	nForms = 0;

	AccountPlans  = NULL;
	AccListForms  = NULL;
	ProvListForms = NULL;
	OperListForms = NULL;
	
	this->BuhID = atoi(BuhID);
}

CMMSObject* CMMSBuh::AddProperty(CMMSObject* Property)
{
	Properties.Add(Property);
	nProps++;

	if( Property->Type == MMS_Plans )
	{
		AccountPlans = Property;
	}
	else if( Property->Type == MMS_OperJournal )
	{
		OperListForms = Property->GetPropByNum(0)->GetProperty(CString("Form"));
		OperListForms->Type = NT_OperListForm;
	}
	else if( Property->sID == "Form" )
	{
		switch( nForms )
		{
			case 0:
				AccListForms = Property;
				AccListForms->Type = NT_AccListForm;
				break;
			case 1:
				UnknownListForms = Property;
				break;
			case 2:
				ProvListForms = Property;
				ProvListForms->Type = NT_ProvListForm;
				break;
		}
		nForms++;
	}

	return Property;
}

CString CMMSBuh::HdrAsString()
{
	CString str;
	str.Format("\"%s\"", sID);
	return str;

}

// true,        .      
bool CMMSBuh::PropsAsString(CString& dest, int Depth)
{
	CString hdr;
	hdr.Format("\r\n{\"%i\",", ID);
	dest += hdr;
	CMMSObject::PropsAsString(dest, Depth);
	dest += "\r\n}";
	return true;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSAccountPlans methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSAccountPlans::CMMSAccountPlans(CString& sID, CMMSNodeList *pList)
:CMMSObjectWithProps(sID, CString(), pList)
{
	Type = MMS_Plans;
	ID_Type = CMMSObject::IDT_OnlyTextual;
}

///////////////////////////////////////////////////////////////////////
///////////  class CMMSAccountPlan methods  ///////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSAccountPlan::CMMSAccountPlan(CString& sID, CString& ID, 
								 CString& Comment, CString& Synonym, 
								 CString& AccountMask1, CString& AccountMask2, 
								 CMMSObject* pAccList)
:CMMSBasicObject(sID, ID, Comment, Synonym, NULL)
{
	Type = MMS_Plan;
	
	//       10009      
	//this->AccountMask1 = AccountMask1;

	this->AccountMask2 = AccountMask2;
	Accounts = pAccList;

	AddProperty(Accounts);
}


CString CMMSAccountPlan::HdrAsString()
{
	CString str;
	if( AccountMask1.IsEmpty() )
	{ // 10009
		str.Format("%s,\"%s\"", CMMSBasicObject::HdrAsString(), AccountMask2);
	}
	else
	{ // 10011
		str.Format("%s,\"%s\",\"%s\"", CMMSBasicObject::HdrAsString(), AccountMask1, AccountMask2);
	}
	return str;
}


///////////////////////////////////////////////////////////////////////
///////////  class CMMSAccount methods  ///////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSAccount::CMMSAccount(CString& sID, CString& ID, CString& Name, CString& Param2, 
		CString& Code, CString& IsCurrency, CString& IsAmount, 
		CString& IsSingle, CString& IsGroup, CString& IsActive,
		CMMSNodeList *pSubcontoList)
:CMMSObjectWithProps(sID, ID, pSubcontoList)
{
	Type = MMS_Account;
	this->Name       = Name;
	this->Param2     = Param2;
	this->Code       = Code;
	this->IsCurrency = atoi(IsCurrency);
	this->IsAmount   = atoi(IsAmount);
	this->IsSingle   = atoi(IsSingle);
	this->IsGroup    = atoi(IsGroup);
	this->IsActive   = atoi(IsActive);

}

CString CMMSAccount::HdrAsString()
{
	CString str;
	
	str.Format("\"%i\",\"%s\",\"%s\",\"%s\",\"%s\",\"%i\",\"%i\",\"%i\",\"%i\",\"%i\"", 
		ID, sID, Name, Param2, Code, 
		IsCurrency, IsAmount, IsSingle, IsGroup, IsActive);
	
	/*
	str.Format("\"\"\r\nID:\t%i %s\r\n:\t%s\r\nParam2:\t%s\r\n:\t%s\r\n:\t%i\r\n:\t%i\r\n:\t%i\r\n:\t%i\r\n:\t%i", 
		ID, sID, Name, Param2, Code, 
		IsCurrency, IsAmount, IsSingle, IsGroup, IsActive);
		*/

	return str;
}



///////////////////////////////////////////////////////////////////////
///////////  class CMMSNode methods  //////////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSNode::~CMMSNode()
{
	if( Properties != NULL )
	{
		delete Properties;
	}
}

void CMMSNode::AddValue(CString& Value)
{
	CMMSNode *NewNode = new CMMSNode(MMS_Property, Value, Value);
	if( Properties == NULL ) Properties = new CMMSNodeList;
	Properties->AddTail((CMMSObject*&)NewNode);
}

void CMMSNode::AddValue(CMMSObject *Node)
{
	if( Properties == NULL ) Properties = new CMMSNodeList;
	Properties->AddTail((CMMSObject*&)Node);
}

int CMMSNode::GetNProps()
{
	if( Properties == NULL ) return 0;
	return Properties->GetCount();
}

CMMSObject* CMMSNode::GetPropByNum(int i)
{
	POSITION pos = Properties->GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* obj = Properties->GetNext(pos);
		if( i <= 0 ) return obj;
		i--;
	}
	return NULL;
}

CString CMMSNode::HdrAsString()
{
	if( sID.GetLength() == 0 ) return ""; //  ,     
	return CString("\"") + sID + "\"";
}

bool CMMSNode::PropsAsString(CString& dest, int Depth)
{
	CString comma(",");
	CString offset("\r\n");
	str_repeate(offset, "\t", Depth+1);
	bool HasObjects = false;

	int i = 0;
	POSITION pos = Properties->GetHeadPosition();
	while( pos != NULL )
	{
		CMMSObject* prop = Properties->GetNext(pos);
		if( i > 0 ) dest += comma;
		if( prop->Type != MMS_Property )
		{
			dest += offset;
			HasObjects = true;
		}
		dest += prop->AsString(Depth+1);
		i++;
	}

	return HasObjects;
}



///////////////////////////////////////////////////////////////////////
///////////  class CMMSNodeList methods  //////////////////////////////
///////////////////////////////////////////////////////////////////////
CMMSNodeList::~CMMSNodeList()
{
	if( DestroyObjects )
	{
		POSITION pos = GetHeadPosition();
		while( pos != NULL )
		{
			CMMSObject* SubNode = GetNext(pos);
			delete SubNode;
		}
	}
}

void CMMSNodeList::AddNode(CString& Value)
{
	CMMSNode *NewNode = new CMMSNode(MMS_Property, Value, Value);
	AddTail((CMMSObject*&)NewNode);
}

void CMMSNodeList::AddNode(CMMSObject *Node)
{
	AddTail((CMMSObject*&)Node);
}

void CMMSNodeList::AddNode(CMMSNode *Node)
{
	AddTail((CMMSObject*&)Node);
}

